/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { take } from 'rxjs/operators';
export class BaseActionsAdapter {
    constructor(actionManager, asyncActionService, message, confirmation, language, selectModalService, metadata, appMetadataStore) {
        this.actionManager = actionManager;
        this.asyncActionService = asyncActionService;
        this.message = message;
        this.confirmation = confirmation;
        this.language = language;
        this.selectModalService = selectModalService;
        this.metadata = metadata;
        this.appMetadataStore = appMetadataStore;
        this.defaultActions = {
            detail: [],
            list: [],
            edit: [],
            create: [],
            massupdate: []
        };
    }
    /**
     * Run the action using given context
     * @param action
     * @param context
     */
    runAction(action, context = null) {
        const params = (action && action.params) || {};
        const displayConfirmation = params.displayConfirmation || false;
        const confirmationLabel = params.confirmationLabel || '';
        const selectModal = action.params && action.params.selectModal;
        const selectModule = selectModal && selectModal.module;
        if (displayConfirmation) {
            this.confirmation.showModal(confirmationLabel, () => {
                if (!selectModule) {
                    this.callAction(action, context);
                    return;
                }
                this.showSelectModal(selectModal.module, action, context);
            });
            return;
        }
        if (!selectModule) {
            this.callAction(action, context);
            return;
        }
        this.showSelectModal(selectModal.module, action, context);
    }
    /**
     * Run async buk action
     *
     * @returns void
     * @param {string} selectModule: module for which records are listed in Select Modal/Popup
     * @param {string} asyncAction: bulk action name
     * @param {ActionContext} context
     */
    showSelectModal(selectModule, asyncAction, context = null) {
        this.selectModalService.showSelectModal(selectModule, (modalRecord) => {
            if (modalRecord) {
                const { fields, formGroup, ...baseRecord } = modalRecord;
                asyncAction.params.modalRecord = baseRecord;
            }
            this.callAction(asyncAction, context);
        });
    }
    /**
     * Get action name
     * @param action
     */
    getActionName(action) {
        return `${action.key}`;
    }
    /**
     * Parse mode actions
     * @param declaredActions
     * @param mode
     * @param context
     */
    parseModeActions(declaredActions, mode, context = null) {
        if (!declaredActions) {
            return [];
        }
        const availableActions = {
            list: [],
            detail: [],
            edit: [],
            create: [],
            massupdate: [],
        };
        if (declaredActions && declaredActions.length) {
            declaredActions.forEach(action => {
                if (!action.modes || !action.modes.length) {
                    return;
                }
                action.modes.forEach(actionMode => {
                    if (!availableActions[actionMode] && !action.asyncProcess) {
                        return;
                    }
                    availableActions[actionMode].push(action);
                });
            });
        }
        availableActions.detail = availableActions.detail.concat(this.defaultActions.detail ?? []);
        availableActions.list = availableActions.list.concat(this.defaultActions.list ?? []);
        availableActions.edit = availableActions.edit.concat(this.defaultActions.edit ?? []);
        availableActions.create = availableActions.create.concat(this.defaultActions.create ?? []);
        availableActions.massupdate = availableActions.massupdate.concat(this.defaultActions.massupdate ?? []);
        const actions = [];
        availableActions[mode].forEach(action => {
            const actionHandler = this.actionManager.getHandler(action, mode);
            if (actionHandler) {
                const data = this.buildActionData(action, context);
                if (!this.shouldDisplay(actionHandler, data)) {
                    return;
                }
                action.status = actionHandler.getStatus(data) || '';
            }
            if (!actionHandler && !action?.asyncProcess) {
                return;
            }
            const module = (context && context.module) || '';
            const label = this.language.getFieldLabel(action.labelKey, module);
            actions.push({
                ...action,
                label
            });
        });
        return actions;
    }
    shouldDisplay(actionHandler, data) {
        return actionHandler && actionHandler.shouldDisplay(data);
    }
    /**
     * Call actions
     * @param action
     * @param context
     */
    callAction(action, context = null) {
        if (action.asyncProcess) {
            this.runAsyncAction(action, context);
            return;
        }
        this.runFrontEndAction(action, context);
    }
    /**
     * Run async actions
     * @param action
     * @param context
     */
    runAsyncAction(action, context = null) {
        const actionName = this.getActionName(action);
        const moduleName = this.getModuleName(context);
        this.message.removeMessages();
        const asyncData = this.buildActionInput(action, actionName, moduleName, context);
        this.asyncActionService.run(actionName, asyncData).pipe(take(1)).subscribe((process) => {
            this.afterAsyncAction(actionName, moduleName, asyncData, process, action, context);
        });
    }
    /**
     * Run after async action handlers
     * @param actionName
     * @param moduleName
     * @param asyncData
     * @param process
     * @param action
     * @param context
     * @protected
     */
    afterAsyncAction(actionName, moduleName, asyncData, process, action, context) {
        if (this.shouldReload(process)) {
            this.reload(action, process, context);
        }
        this.reloadMetadata(moduleName, action, process, context);
    }
    /**
     * Reload the metadata for the module
     * @param moduleName
     * @param action
     * @param process
     * @param context
     * @protected
     */
    reloadMetadata(moduleName, action, process, context) {
        const typesToLoad = [];
        if (this.shouldReloadRecentlyViewed(process)) {
            typesToLoad.push(this.metadata.typeKeys.recentlyViewed);
        }
        if (this.shouldReloadFavorites(process)) {
            typesToLoad.push(this.metadata.typeKeys.favorites);
        }
        if (typesToLoad && typesToLoad.length) {
            this.metadata.reloadModuleMetadata(moduleName, typesToLoad, false).pipe(take(1)).subscribe();
            if (typesToLoad.includes(this.metadata.typeKeys.recentlyViewed)) {
                this.appMetadataStore.load(moduleName, ['globalRecentlyViewed'], false).pipe(take(1)).subscribe();
            }
        }
    }
    /**
     * Should reload page
     * @param process
     */
    shouldReloadRecentlyViewed(process) {
        return !!(process.data && process.data.reloadRecentlyViewed);
    }
    /**
     * Should reload page
     * @param process
     */
    shouldReloadFavorites(process) {
        return !!(process.data && process.data.reloadFavorites);
    }
    /**
     * Should reload page
     * @param process
     */
    shouldReload(process) {
        return !!(process.data && process.data.reload);
    }
    /**
     * Run front end action
     * @param action
     * @param context
     */
    runFrontEndAction(action, context = null) {
        const data = this.buildActionData(action, context);
        this.actionManager.run(action, this.getMode(), data);
    }
}
//# sourceMappingURL=data:application/json;base64,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