/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { isArray, isEmpty, union } from 'lodash-es';
import { deepClone, emptyObject, isTrue } from 'common';
import { BehaviorSubject, combineLatestWith } from 'rxjs';
import { distinctUntilChanged, map, take } from 'rxjs/operators';
import { Injectable } from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { NavigationStore } from '../../../../store/navigation/navigation.store';
import { MetadataStore } from '../../../../store/metadata/metadata.store.service';
import { LanguageStore } from '../../../../store/language/language.store';
import { ModuleNavigation } from '../../../../services/navigation/module-navigation/module-navigation.service';
import { MessageService } from '../../../../services/message/message.service';
import { RecordListStoreFactory } from '../../../../store/record-list/record-list.store.factory';
import { AppStateStore } from '../../../../store/app-state/app-state.store';
import { ViewStore } from '../../../../store/view/view.store';
import { LocalStorageService } from '../../../../services/local-storage/local-storage.service';
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { ColumnChooserComponent } from "../../../../components/columnchooser/columnchooser.component";
import { FilterListStoreFactory } from '../../../../store/saved-filters/filter-list.store.factory';
import { ConfirmationModalService } from '../../../../services/modals/confirmation-modal.service';
import { UserPreferenceStore } from '../../../../store/user-preference/user-preference.store';
import { ListViewUrlQueryService } from '../../services/list-view-url-query.service';
import { SystemConfigStore } from "../../../../store/system-config/system-config.store";
import * as i0 from "@angular/core";
import * as i1 from "../../../../store/app-state/app-state.store";
import * as i2 from "../../../../store/language/language.store";
import * as i3 from "../../../../store/navigation/navigation.store";
import * as i4 from "../../../../services/navigation/module-navigation/module-navigation.service";
import * as i5 from "../../../../store/metadata/metadata.store.service";
import * as i6 from "../../../../services/message/message.service";
import * as i7 from "../../../../store/record-list/record-list.store.factory";
import * as i8 from "@ng-bootstrap/ng-bootstrap";
import * as i9 from "../../../../store/saved-filters/filter-list.store.factory";
import * as i10 from "../../../../services/modals/confirmation-modal.service";
import * as i11 from "../../../../store/user-preference/user-preference.store";
import * as i12 from "@angular/router";
import * as i13 from "../../services/list-view-url-query.service";
import * as i14 from "../../../../services/local-storage/local-storage.service";
import * as i15 from "../../../../store/system-config/system-config.store";
const initialFilter = {
    key: 'default',
    module: 'saved-search',
    attributes: {
        contents: ''
    },
    criteria: {
        name: 'default',
        filters: {}
    }
};
const initialFilters = {
    'default': deepClone(initialFilter)
};
const initialState = {
    module: '',
    widgets: true,
    actionPanel: '',
    showSidebarWidgets: false,
    recordPanelConfig: {},
    activeFilters: deepClone(initialFilters),
    openFilter: deepClone(initialFilter)
};
class ListViewStore extends ViewStore {
    constructor(appStateStore, languageStore, navigationStore, moduleNavigation, metadataStore, message, listStoreFactory, modalService, filterListStoreFactory, confirmation, preferences, route, listViewUrlQueryService, localStorageService, systemConfigsStore, userPreferences) {
        super(appStateStore, languageStore, navigationStore, moduleNavigation, metadataStore);
        this.appStateStore = appStateStore;
        this.languageStore = languageStore;
        this.navigationStore = navigationStore;
        this.moduleNavigation = moduleNavigation;
        this.metadataStore = metadataStore;
        this.message = message;
        this.listStoreFactory = listStoreFactory;
        this.modalService = modalService;
        this.filterListStoreFactory = filterListStoreFactory;
        this.confirmation = confirmation;
        this.preferences = preferences;
        this.route = route;
        this.listViewUrlQueryService = listViewUrlQueryService;
        this.localStorageService = localStorageService;
        this.systemConfigsStore = systemConfigsStore;
        this.userPreferences = userPreferences;
        this.pageKey = 'listview';
        /** Internal Properties */
        this.cache$ = null;
        this.internalState = deepClone(initialState);
        this.store = new BehaviorSubject(this.internalState);
        this.state$ = this.store.asObservable();
        this.subs = [];
        this.recordList = this.listStoreFactory.create();
        this.columns$ = metadataStore.listViewColumns$;
        this.lineActions$ = metadataStore.listViewLineActions$;
        this.tableActions$ = metadataStore.listViewTableActions$;
        this.records$ = this.recordList.records$;
        this.criteria$ = this.recordList.criteria$;
        this.context$ = this.recordList.criteria$.pipe(map(() => this.getViewContext()));
        this.sort$ = this.recordList.sort$;
        this.pagination$ = this.recordList.pagination$;
        this.selection$ = this.recordList.selection$;
        this.selectedCount$ = this.recordList.selectedCount$;
        this.selectedStatus$ = this.recordList.selectedStatus$;
        this.loading$ = this.recordList.loading$;
        this.moduleName$ = this.state$.pipe(map(state => state.module), distinctUntilChanged());
        this.widgets$ = this.state$.pipe(map(state => state.widgets), distinctUntilChanged());
        this.showSidebarWidgets$ = this.state$.pipe(map(state => state.showSidebarWidgets));
        this.displayFilters$ = this.state$.pipe(map(state => state.actionPanel === 'filters'), distinctUntilChanged());
        this.actionPanel$ = this.state$.pipe(map(state => state.actionPanel), distinctUntilChanged());
        this.activeFilters$ = this.state$.pipe(map(state => state.activeFilters), distinctUntilChanged());
        this.openFilter$ = this.state$.pipe(map(state => state.openFilter), distinctUntilChanged());
        const data$ = this.records$.pipe(combineLatestWith(this.criteria$, this.pagination$, this.selection$, this.loading$), map(([records, criteria, pagination, selection, loading]) => {
            this.data = { records, criteria, pagination, selection, loading };
            return this.data;
        }));
        this.vm$ = data$.pipe(combineLatestWith(this.appData$, this.metadata$), map(([data, appData, metadata]) => {
            this.vm = { data, appData, metadata };
            return this.vm;
        }));
        this.columns = new BehaviorSubject([]);
        this.columns$ = this.columns.asObservable();
        this.initDataUpdateState();
        this.initDataSetUpdatedState();
        this.filterList = this.filterListStoreFactory.create();
        this.recordList.pageKey = this.pageKey;
    }
    get actionPanel() {
        return this.internalState.actionPanel;
    }
    get showFilters() {
        return this.internalState.actionPanel === 'filters';
    }
    set showFilters(show) {
        this.updateState({
            ...this.internalState,
            actionPanel: show ? 'filters' : ''
        });
    }
    get widgets() {
        return this.internalState.widgets;
    }
    set widgets(show) {
        this.updateState({
            ...this.internalState,
            widgets: show
        });
    }
    get showSidebarWidgets() {
        return this.internalState.showSidebarWidgets;
    }
    set showSidebarWidgets(show) {
        this.savePreference(this.getModuleName(), 'show-sidebar-widgets', show);
        this.updateState({
            ...this.internalState,
            showSidebarWidgets: show
        });
    }
    get recordPanelConfig() {
        return this.internalState.recordPanelConfig;
    }
    isRecordPanelOpen() {
        return this.internalState.actionPanel === 'recordPanel';
    }
    openRecordPanel(config) {
        this.updateState({
            ...this.internalState,
            actionPanel: 'recordPanel',
            recordPanelConfig: config
        });
    }
    closeRecordPanel() {
        this.updateState({
            ...this.internalState,
            actionPanel: '',
            recordPanelConfig: {}
        });
    }
    getModuleName() {
        return this.internalState.module;
    }
    getViewContext() {
        const context = {
            module: this.getModuleName(),
        };
        context.criteria = this.recordList.criteria;
        context.sort = this.recordList.sort;
        return context;
    }
    /**
     * Clean destroy
     */
    destroy() {
        this.clear();
        this.subs.forEach(sub => sub.unsubscribe());
    }
    /**
     * get active filters
     *
     * @returns {object} active filters
     */
    get activeFilters() {
        return deepClone(this.internalState.activeFilters);
    }
    /**
     * Clear observable cache
     */
    clear() {
        this.cache$ = null;
        this.updateState(deepClone(initialState));
        this.recordList.clear();
    }
    clearAuthBased() {
        this.clear();
        this.recordList.clearAuthBased();
    }
    /**
     * Initial list records load if not cached and update state.
     * Returns observable to be used in resolver if needed
     *
     * @param {string} module to use
     * @returns {object} Observable<any>
     */
    init(module) {
        this.internalState.module = module;
        this.recordList.init(module, false);
        this.filterList.init(module);
        this.filterList.load(false).pipe(take(1)).subscribe();
        this.calculateShowWidgets();
        this.recordList.sort = {
            orderBy: this?.metadata?.listView?.orderBy ?? '',
            sortOrder: this?.metadata?.listView?.sortOrder ?? 'NONE'
        };
        const queryParams = this.route?.snapshot?.queryParams ?? {};
        let filterType = '';
        if (isTrue(queryParams['query'])) {
            filterType = 'query';
        }
        switch (filterType) {
            case 'query':
                this.loadQueryFilter(module, queryParams);
                break;
            default:
                this.loadCurrentFilter(module);
                this.loadCurrentSort(module);
        }
        this.loadCurrentDisplayedColumns();
        const paginationType = this.userPreferences.getUserPreference('listview_pagination_type') ?? this.systemConfigsStore.getConfigValue('listview_pagination_type');
        const currentPaginationType = this.getCurrentPaginationType(module);
        this.setCurrentPaginationType(module, paginationType);
        if (queryParams['keepPagination'] && currentPaginationType === paginationType) {
            this.loadCurrentPagination(module);
        }
        return this.load();
    }
    /**
     * Set open filters
     *
     * @param {object} filter to set
     */
    setOpenFilter(filter) {
        this.updateState({ ...this.internalState, openFilter: deepClone(filter) });
    }
    /**
     * Toggle Quick filter
     *
     * @param filter
     * @param {boolean} reload flag
     */
    toggleQuickFilter(filter, reload = true) {
        let activeFilters = this.getActiveQuickFilters();
        const isActive = Object.keys(activeFilters).some(key => key === filter.key);
        if (isActive) {
            let { [filter.key]: removedFilters, ...newFilters } = activeFilters;
            activeFilters = newFilters;
        }
        else {
            activeFilters = {};
            activeFilters[filter.key] = filter;
        }
        if (emptyObject(activeFilters)) {
            this.resetFilters(reload);
            return;
        }
        if (Object.keys(activeFilters).length === 1) {
            this.setFilters(activeFilters);
            return;
        }
        this.updateState({
            ...this.internalState,
            activeFilters: deepClone(activeFilters),
        });
        this.updateSearchCriteria(reload);
    }
    /**
     * Set active filters
     *
     * @param {object} filters to set
     * @param {boolean} reload flag
     * @param sort
     */
    setFilters(filters, reload = true, sort = null) {
        const filterKey = Object.keys(filters).shift();
        const filter = filters[filterKey];
        this.updateState({ ...this.internalState, activeFilters: deepClone(filters), openFilter: deepClone(filter) });
        if (filter.criteria) {
            let orderBy = filter.criteria.orderBy ?? '';
            const sortOrder = filter.criteria.sortOrder ?? '';
            let direction = this.recordList.mapSortOrder(sortOrder);
            if (sort !== null) {
                orderBy = sort.orderBy;
                direction = sort.sortOrder;
            }
            this.recordList.updateSorting(orderBy, direction, false);
            this.updateSortLocalStorage();
        }
        this.updateSearchCriteria(reload);
    }
    /**
     * Update filters
     *
     * @param {object} filter to set
     */
    addSavedFilter(filter) {
        const newState = { ...this.internalState };
        const activeFilters = this.activeFilters;
        if (filter.key && activeFilters[filter.key]) {
            activeFilters[filter.key] = filter;
            newState.activeFilters = activeFilters;
        }
        newState.openFilter = filter;
        this.filterList.addFilter(filter);
        this.updateState(newState);
    }
    /**
     * Update filters
     *
     * @param {object} filter to set
     */
    removeSavedFilter(filter) {
        if (!filter || !filter.key) {
            return;
        }
        this.filterList.removeFilter(filter);
        const newState = { ...this.internalState };
        if (newState.openFilter && newState.openFilter.key === filter.key) {
            this.resetFilters(true);
        }
    }
    /**
     * Reset active filters
     *
     * @param {boolean} reload flag
     */
    resetFilters(reload = true) {
        this.updateState({
            ...this.internalState,
            activeFilters: deepClone(initialFilters),
            openFilter: deepClone(initialFilter),
        });
        this.recordList.clearSort();
        this.updateSortLocalStorage();
        this.updateSearchCriteria(reload);
    }
    /**
     * Update the search criteria
     *
     * @param {boolean} reload flag
     */
    updateSearchCriteria(reload = true) {
        const filters = { ...this.internalState.activeFilters };
        let criteria = this.mergeCriteria(filters);
        this.recordList.updateSearchCriteria(criteria, reload);
        this.updateFilterLocalStorage();
    }
    updateFilterLocalStorage() {
        const module = this.internalState.module;
        this.savePreference(module, 'current-filters', this.internalState.activeFilters);
    }
    updateSortLocalStorage() {
        const module = this.internalState.module;
        this.savePreference(module, 'current-sort', this.recordList.sort);
    }
    updatePaginationLocalStorage() {
        const module = this.internalState.module;
        const key = module + '-' + this.getPreferenceKey('current-pagination');
        this.localStorageService.set(key, this.recordList.pagination);
    }
    /**
     * Updated displayed columns' ui user preference
     * @param display
     */
    updateDisplayedColumnsPreference(display) {
        const module = this.internalState.module;
        this.savePreference(module, 'displayed-columns', display);
    }
    /**
     * Get displayed columns' ui user preference
     */
    getDisplayedColumnsPreference() {
        const module = this.internalState.module;
        const displayedColumns = this.loadPreference(module, 'displayed-columns');
        if (!isArray(displayedColumns) || !displayedColumns || !displayedColumns.length) {
            return null;
        }
        return displayedColumns;
    }
    triggerDataUpdate() {
        this.dataUpdateState.next(true);
    }
    /**
     * Load / reload records using current pagination and criteria
     *
     * @param {boolean} useCache if to use cache
     * @returns {object} Observable<ListViewState>
     */
    load(useCache = true) {
        const module = this.internalState.module;
        this.savePreference(module, 'current-filters', this.internalState.activeFilters);
        this.savePreference(module, 'current-sort', this.recordList.sort);
        this.updatePaginationLocalStorage();
        return this.recordList.load(useCache);
    }
    /**
     * Internal API
     */
    /**
     * Update the state
     *
     * @param {object} state to set
     */
    updateState(state) {
        this.store.next(this.internalState = state);
    }
    /**
     * Get Active quick filters
     * @protected
     */
    getActiveQuickFilters() {
        let { 'default': defaultFilter, ...currentQuickFilters } = this.activeFilters;
        let activeFilters = {};
        Object.keys(currentQuickFilters).forEach(key => {
            const activeFilter = currentQuickFilters[key] ?? null;
            if (!key) {
                return;
            }
            const isQuickFilter = activeFilter?.attributes?.quick_filter ?? false;
            if (isQuickFilter) {
                activeFilters[key] = activeFilter;
            }
        });
        return activeFilters;
    }
    /**
     * Merge Criteria
     * @protected
     */
    mergeCriteria(filters) {
        let criteria = {};
        const keys = Object.keys(filters ?? {}) ?? [];
        keys.forEach(key => {
            const filter = filters[key] ?? null;
            const filterCriteria = filter?.criteria ?? null;
            const filterCriteriaKeys = Object.keys(filterCriteria?.filters ?? {});
            if (filterCriteria === null || (filterCriteriaKeys && !filterCriteriaKeys.length)) {
                return;
            }
            if (emptyObject(criteria)) {
                criteria = deepClone(filterCriteria);
                return;
            }
            filterCriteriaKeys.forEach(criteriaKey => {
                const filterCriteriaContent = filterCriteria?.filters[criteriaKey] ?? null;
                const criteriaContent = criteria?.filters[criteriaKey] ?? null;
                if (!filterCriteriaContent) {
                    return;
                }
                const criteriaOperator = criteriaContent?.operator ?? null;
                if (!criteriaContent || !criteriaOperator) {
                    criteria.filters[criteriaKey] = deepClone(filterCriteriaContent);
                    return;
                }
                const filterCriteriaOperator = filterCriteriaContent?.operator ?? null;
                if (filterCriteriaOperator !== criteriaOperator || filterCriteriaOperator !== '=') {
                    delete criteria.filters[criteriaKey];
                    return;
                }
                criteriaContent.values = union(criteriaContent.values ?? [], filterCriteriaContent.values ?? []);
            });
        });
        return criteria;
    }
    /**
     * Open columns chooser modal
     */
    openColumnChooserDialog() {
        const modalRef = this.modalService.open(ColumnChooserComponent, {
            ariaLabelledBy: 'modal-basic-title',
            centered: true,
            size: 'lg',
            windowClass: 'column-chooser-modal'
        });
        const displayedColumns = this.columns.getValue().filter(function (col) {
            return !col.hasOwnProperty('default')
                || (col.hasOwnProperty('default') && col.default === true);
        });
        const hiddenColumns = this.columns.getValue().filter(function (col) {
            return col.hasOwnProperty('default') && col.default === false;
        });
        modalRef.componentInstance.displayed = displayedColumns;
        modalRef.componentInstance.hidden = hiddenColumns;
        modalRef.result.then((result) => {
            if (!result.displayed || !result.hidden) {
                return;
            }
            let allColumns = [];
            const selectedDisplayColumns = result.displayed;
            const selectedHideColumns = result.hidden;
            selectedDisplayColumns.forEach(function (column) {
                column.default = true;
            });
            selectedHideColumns.forEach(function (column) {
                column.default = false;
            });
            allColumns.push(...selectedDisplayColumns, ...selectedHideColumns);
            this.columns.next(allColumns);
            const displayedCols = selectedDisplayColumns.map(col => col.name);
            this.updateDisplayedColumnsPreference(displayedCols);
        });
    }
    /**
     * Calculate if widgets are to display
     */
    calculateShowWidgets() {
        let show = false;
        const meta = this.metadataStore.get() || {};
        const listViewMeta = meta.listView || {};
        const sidebarWidgetsConfig = listViewMeta.sidebarWidgets || [];
        if (sidebarWidgetsConfig && sidebarWidgetsConfig.length > 0) {
            show = true;
        }
        const showSidebarWidgets = this.loadPreference(this.getModuleName(), 'show-sidebar-widgets') ?? null;
        if (showSidebarWidgets !== null) {
            this.showSidebarWidgets = showSidebarWidgets;
        }
        else {
            this.showSidebarWidgets = show;
        }
        this.widgets = show;
    }
    /**
     * Build ui user preference key
     * @param storageKey
     * @protected
     */
    getPreferenceKey(storageKey) {
        return this.pageKey + '-' + storageKey;
    }
    /**
     * Save ui user preference
     * @param module
     * @param storageKey
     * @param value
     * @protected
     */
    savePreference(module, storageKey, value) {
        this.preferences.setUi(module, this.getPreferenceKey(storageKey), value);
    }
    /**
     * Load ui user preference
     * @param module
     * @param storageKey
     * @protected
     */
    loadPreference(module, storageKey) {
        return this.preferences.getUi(module, this.getPreferenceKey(storageKey));
    }
    /**
     * Load current filter
     * @param module
     * @protected
     */
    loadCurrentFilter(module) {
        const activeFiltersPref = this.loadPreference(module, 'current-filters') ?? {};
        if (!activeFiltersPref || emptyObject(activeFiltersPref)) {
            return;
        }
        let currentSort = this.loadPreference(module, 'current-sort');
        if (!currentSort && emptyObject(currentSort)) {
            currentSort = null;
        }
        this.setFilters(activeFiltersPref, false, currentSort);
    }
    /**
     * Load current filter
     * @param module
     * @param queryParams
     * @protected
     */
    loadQueryFilter(module, queryParams) {
        const orderBy = queryParams['orderBy'] ?? '';
        const sortOrder = queryParams['sortOrder'] ?? '';
        const direction = this.recordList.mapSortOrder(sortOrder);
        const filter = this.listViewUrlQueryService.buildUrlQueryBasedFilter(module, this.internalState.activeFilters.default, queryParams);
        if (isEmpty(filter)) {
            return;
        }
        const filters = { 'default': filter };
        this.updateState({
            ...this.internalState,
            activeFilters: deepClone(filters),
            openFilter: deepClone(filter)
        });
        this.recordList.updateSorting(orderBy, direction, false);
        this.recordList.updateSearchCriteria(filter.criteria, false);
    }
    /**
     * Load current sorting
     * @param module
     * @protected
     */
    loadCurrentSort(module) {
        const currentSort = this.loadPreference(module, 'current-sort');
        if (!currentSort || emptyObject(currentSort)) {
            return;
        }
        this.recordList.sort = currentSort;
    }
    /**
     * Load current pagination
     * @param module
     * @protected
     */
    loadCurrentPagination(module) {
        const key = module + '-' + this.getPreferenceKey('current-pagination');
        const currentPagination = this.localStorageService.get(key);
        if (!currentPagination || emptyObject(currentPagination)) {
            return;
        }
        this.recordList.pagination = currentPagination;
    }
    /**
     * Get current pagination Type
     * @param module
     * @protected
     */
    getCurrentPaginationType(module) {
        const currentPaginationType = this.loadPreference(module, 'current-pagination-type');
        if (!currentPaginationType) {
            return 'pagination';
        }
        return currentPaginationType;
    }
    /**
     * Set current pagination Type
     * @param module
     * @protected
     */
    setCurrentPaginationType(module, paginationType) {
        this.savePreference(module, 'current-pagination-type', paginationType);
    }
    /**
     * Load current displayed columns
     * @protected
     */
    loadCurrentDisplayedColumns() {
        this.metadataStore.listViewColumns$.pipe(take(1)).subscribe(cols => {
            const displayedColumns = this.getDisplayedColumnsPreference();
            if (!displayedColumns || !cols) {
                this.columns.next(cols);
                return;
            }
            const colMap = {};
            displayedColumns.forEach(displayedColumn => {
                colMap[displayedColumn] = true;
            });
            const displayedMap = {};
            const hidden = [];
            cols.forEach(col => {
                col.default = colMap[col.name] ?? false;
                if (col.default) {
                    displayedMap[col.name] = col;
                }
                else {
                    hidden.push(col);
                }
            });
            const displayed = displayedColumns.filter(col => !!displayedMap[col]).map(col => displayedMap[col]);
            this.columns.next([...displayed, ...hidden]);
        });
    }
    /**
     * Initialize data update state.
     * It should be emitted on any change in values on the record list.
     * Reload/Pagination is not considered as a data update
     */
    initDataUpdateState() {
        this.dataUpdateState = new BehaviorSubject(true);
        this.dataUpdate$ = this.dataUpdateState.asObservable();
    }
    /**
     *  Initialize the dataSet update state.
     *  It should be emitted on any change in dataSet e.g. due to data filter, due to data delete,
     *  due to data edit or any event which causes change in the resulting dataSet.
     */
    initDataSetUpdatedState() {
        this.dataSetUpdate$ = this.criteria$.pipe(combineLatestWith(this.dataUpdate$), map(() => true));
    }
    static { this.ɵfac = function ListViewStore_Factory(t) { return new (t || ListViewStore)(i0.ɵɵinject(i1.AppStateStore), i0.ɵɵinject(i2.LanguageStore), i0.ɵɵinject(i3.NavigationStore), i0.ɵɵinject(i4.ModuleNavigation), i0.ɵɵinject(i5.MetadataStore), i0.ɵɵinject(i6.MessageService), i0.ɵɵinject(i7.RecordListStoreFactory), i0.ɵɵinject(i8.NgbModal), i0.ɵɵinject(i9.FilterListStoreFactory), i0.ɵɵinject(i10.ConfirmationModalService), i0.ɵɵinject(i11.UserPreferenceStore), i0.ɵɵinject(i12.ActivatedRoute), i0.ɵɵinject(i13.ListViewUrlQueryService), i0.ɵɵinject(i14.LocalStorageService), i0.ɵɵinject(i15.SystemConfigStore), i0.ɵɵinject(i11.UserPreferenceStore)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ListViewStore, factory: ListViewStore.ɵfac }); }
}
export { ListViewStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ListViewStore, [{
        type: Injectable
    }], function () { return [{ type: i1.AppStateStore }, { type: i2.LanguageStore }, { type: i3.NavigationStore }, { type: i4.ModuleNavigation }, { type: i5.MetadataStore }, { type: i6.MessageService }, { type: i7.RecordListStoreFactory }, { type: i8.NgbModal }, { type: i9.FilterListStoreFactory }, { type: i10.ConfirmationModalService }, { type: i11.UserPreferenceStore }, { type: i12.ActivatedRoute }, { type: i13.ListViewUrlQueryService }, { type: i14.LocalStorageService }, { type: i15.SystemConfigStore }, { type: i11.UserPreferenceStore }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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