/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { FieldBuilder } from './field.builder';
import { Injectable } from '@angular/core';
import { ValidationManager } from '../validation/validation.manager';
import { DataTypeFormatter } from '../../formatters/data-type.formatter.service';
import { deepClone } from 'common';
import { FieldObjectRegistry } from "./field-object-type.registry";
import * as i0 from "@angular/core";
import * as i1 from "../validation/validation.manager";
import * as i2 from "../../formatters/data-type.formatter.service";
import * as i3 from "./field-object-type.registry";
class FilterFieldBuilder extends FieldBuilder {
    constructor(validationManager, typeFormatter, fieldRegistry) {
        super(validationManager, typeFormatter, fieldRegistry);
        this.validationManager = validationManager;
        this.typeFormatter = typeFormatter;
        this.fieldRegistry = fieldRegistry;
    }
    /**
     * Build filter field
     *
     * @param {object} savedFilter SavedFilter
     * @param {object} viewField ViewFieldDefinition
     * @param {object} language LanguageStore
     * @returns {object} Field
     */
    buildFilterField(savedFilter, viewField, language = null) {
        const definition = (viewField && viewField.fieldDefinition) || {};
        const { validators, asyncValidators } = this.getFilterValidators(savedFilter, viewField);
        const field = this.setupField(savedFilter.searchModule, viewField, null, null, null, savedFilter, definition, validators, asyncValidators, language);
        field.criteria = this.initFieldFilter(savedFilter.criteria, viewField, field);
        return field;
    }
    /**
     * Get Filter Validators for given field definition
     *
     * @param {object} record  Record
     * @param {object} viewField ViewFieldDefinition
     * @returns {object} validator map
     */
    getFilterValidators(record, viewField) {
        const validators = this.validationManager.getFilterValidations(record.searchModule, viewField, record);
        const asyncValidators = [];
        return { validators, asyncValidators };
    }
    /**
     * Init filter fields
     *
     * @param {object} searchCriteria to use
     * @param {object} viewField to init
     * @param {object} field to init
     * @returns {object} SearchCriteriaFieldFilter
     */
    initFieldFilter(searchCriteria, viewField, field) {
        let fieldCriteria;
        const fieldName = viewField.name;
        let fieldType = viewField.type;
        let rangeSearch = false;
        if (fieldType === 'composite') {
            fieldType = field.definition.type;
            rangeSearch = true;
        }
        if (searchCriteria.filters[fieldName] && searchCriteria.filters[fieldName].fieldType) {
            fieldCriteria = deepClone(searchCriteria.filters[fieldName]);
        }
        else {
            fieldCriteria = {
                field: fieldName,
                fieldType,
                operator: '',
                values: []
            };
        }
        fieldCriteria.rangeSearch = rangeSearch;
        this.mapEnumEmptyOption(fieldCriteria, field);
        this.mapRelateFields(fieldCriteria, field, searchCriteria);
        return fieldCriteria;
    }
    mapEnumEmptyOption(fieldCriteria, field) {
        if (!['multienum', 'enum'].includes(fieldCriteria.fieldType)) {
            return;
        }
        let emptyOption = this.getEmptyOption(field);
        if (!emptyOption) {
            return;
        }
        if (!fieldCriteria.values || !fieldCriteria.values.length) {
            return;
        }
        fieldCriteria.values = fieldCriteria.values.map(value => {
            if (value !== '') {
                return value;
            }
            return '__SuiteCRMEmptyString__';
        });
    }
    mapRelateFields(fieldCriteria, field, searchCriteria) {
        if (!['relate'].includes(fieldCriteria.fieldType)) {
            return;
        }
        if (!fieldCriteria.values || !fieldCriteria.values.length) {
            return;
        }
        const idFieldName = (field && field.definition && field.definition.id_name) || '';
        const relateFieldName = (field && field.definition && field.definition.rname) || 'name';
        const idValues = searchCriteria?.filters[idFieldName]?.values ?? [];
        fieldCriteria.valueObjectArray = fieldCriteria.valueObjectArray ?? [];
        const relateFieldMap = {};
        if (fieldCriteria.valueObjectArray.length) {
            fieldCriteria.valueObjectArray.forEach(value => {
                relateFieldMap[value.id] = value;
            });
        }
        for (let i = 0; i < fieldCriteria.values.length; i++) {
            let value = fieldCriteria.values[i];
            const relateValue = {};
            relateValue[relateFieldName] = value;
            relateValue['id'] = idValues[i] ?? '';
            if (!relateFieldMap[relateValue['id']]) {
                relateFieldMap[relateValue['id']] = relateValue;
                fieldCriteria.valueObjectArray.push(relateValue);
            }
        }
    }
    getEmptyOption(field) {
        let emptyOption = null;
        const extraOptions = field?.definition?.metadata?.extraOptions ?? [];
        extraOptions.forEach(option => {
            if (option.value === '__SuiteCRMEmptyString__') {
                emptyOption = option;
            }
        });
        return emptyOption;
    }
    /**
     * Is criteria field initialized in record
     *
     * @param {object} record Record
     * @param {string} fieldName field
     * @returns {boolean} isInitialized
     */
    isCriteriaFieldInitialized(record, fieldName) {
        const criteriaField = record.criteriaFields[fieldName];
        return !!criteriaField && !criteriaField.vardefBased;
    }
    /**
     * Add field to SavedFilter
     *
     * @param {object} savedFilter SavedFilter
     * @param {string} name string
     * @param {object} field Field
     */
    addToSavedFilter(savedFilter, name, field) {
        if (!savedFilter || !name || !field) {
            return;
        }
        if (!savedFilter.criteriaFields) {
            savedFilter.criteriaFields = {};
        }
        savedFilter.criteriaFields[name] = field;
        if (!savedFilter.criteria.filters) {
            savedFilter.criteria.filters = {};
        }
        savedFilter.criteria.filters[name] = field.criteria;
        if (savedFilter.criteriaFormGroup && field.formControl) {
            savedFilter.criteriaFormGroup.addControl(name, field.formControl);
        }
    }
    static { this.ɵfac = function FilterFieldBuilder_Factory(t) { return new (t || FilterFieldBuilder)(i0.ɵɵinject(i1.ValidationManager), i0.ɵɵinject(i2.DataTypeFormatter), i0.ɵɵinject(i3.FieldObjectRegistry)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: FilterFieldBuilder, factory: FilterFieldBuilder.ɵfac, providedIn: 'root' }); }
}
export { FilterFieldBuilder };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(FilterFieldBuilder, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], function () { return [{ type: i1.ValidationManager }, { type: i2.DataTypeFormatter }, { type: i3.FieldObjectRegistry }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmlsdGVyLWZpZWxkLmJ1aWxkZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi9jb3JlL2FwcC9jb3JlL3NyYy9saWIvc2VydmljZXMvcmVjb3JkL2ZpZWxkL2ZpbHRlci1maWVsZC5idWlsZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F3Qkc7QUFFSCxPQUFPLEVBQUMsWUFBWSxFQUFDLE1BQU0saUJBQWlCLENBQUM7QUFDN0MsT0FBTyxFQUFDLFVBQVUsRUFBQyxNQUFNLGVBQWUsQ0FBQztBQUN6QyxPQUFPLEVBQUMsaUJBQWlCLEVBQUMsTUFBTSxrQ0FBa0MsQ0FBQztBQUNuRSxPQUFPLEVBQUMsaUJBQWlCLEVBQUMsTUFBTSw4Q0FBOEMsQ0FBQztBQUUvRSxPQUFPLEVBQ0gsU0FBUyxFQU9aLE1BQU0sUUFBUSxDQUFDO0FBR2hCLE9BQU8sRUFBQyxtQkFBbUIsRUFBQyxNQUFNLDhCQUE4QixDQUFDOzs7OztBQUVqRSxNQUdhLGtCQUFtQixTQUFRLFlBQVk7SUFFaEQsWUFDYyxpQkFBb0MsRUFDcEMsYUFBZ0MsRUFDaEMsYUFBa0M7UUFFNUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLGFBQWEsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUo3QyxzQkFBaUIsR0FBakIsaUJBQWlCLENBQW1CO1FBQ3BDLGtCQUFhLEdBQWIsYUFBYSxDQUFtQjtRQUNoQyxrQkFBYSxHQUFiLGFBQWEsQ0FBcUI7SUFHaEQsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxnQkFBZ0IsQ0FBQyxXQUF3QixFQUFFLFNBQThCLEVBQUUsV0FBMEIsSUFBSTtRQUU1RyxNQUFNLFVBQVUsR0FBRyxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLElBQUksRUFBcUIsQ0FBQztRQUNyRixNQUFNLEVBQUMsVUFBVSxFQUFFLGVBQWUsRUFBQyxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFdkYsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FDekIsV0FBVyxDQUFDLFlBQVksRUFDeEIsU0FBUyxFQUNULElBQUksRUFDSixJQUFJLEVBQ0osSUFBSSxFQUNKLFdBQVcsRUFDWCxVQUFVLEVBQ1YsVUFBVSxFQUNWLGVBQWUsRUFDZixRQUFRLENBQ1gsQ0FBQztRQUNGLEtBQUssQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM5RSxPQUFPLEtBQUssQ0FBQztJQUNqQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CLENBQ3RCLE1BQW1CLEVBQ25CLFNBQThCO1FBRzlCLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN2RyxNQUFNLGVBQWUsR0FBdUIsRUFBRSxDQUFDO1FBRS9DLE9BQU8sRUFBQyxVQUFVLEVBQUUsZUFBZSxFQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxlQUFlLENBQUMsY0FBOEIsRUFBRSxTQUE4QixFQUFFLEtBQVk7UUFDL0YsSUFBSSxhQUF3QyxDQUFDO1FBRTdDLE1BQU0sU0FBUyxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUM7UUFDakMsSUFBSSxTQUFTLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQztRQUMvQixJQUFJLFdBQVcsR0FBRyxLQUFLLENBQUM7UUFFeEIsSUFBSSxTQUFTLEtBQUssV0FBVyxFQUFFO1lBQzNCLFNBQVMsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztZQUNsQyxXQUFXLEdBQUcsSUFBSSxDQUFDO1NBQ3RCO1FBRUQsSUFBSSxjQUFjLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLGNBQWMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsU0FBUyxFQUFFO1lBQ2xGLGFBQWEsR0FBRyxTQUFTLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1NBQ2hFO2FBQU07WUFDSCxhQUFhLEdBQUc7Z0JBQ1osS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLFNBQVM7Z0JBQ1QsUUFBUSxFQUFFLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLEVBQUU7YUFDZ0IsQ0FBQztTQUNsQztRQUVELGFBQWEsQ0FBQyxXQUFXLEdBQUcsV0FBVyxDQUFDO1FBRXhDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFOUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTNELE9BQU8sYUFBYSxDQUFDO0lBQ3pCLENBQUM7SUFFUyxrQkFBa0IsQ0FBQyxhQUF3QyxFQUFFLEtBQVk7UUFDL0UsSUFBSSxDQUFDLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDMUQsT0FBTztTQUNWO1FBRUQsSUFBSSxXQUFXLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUU3QyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ2QsT0FBTztTQUNWO1FBRUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRTtZQUN2RCxPQUFPO1NBQ1Y7UUFFRCxhQUFhLENBQUMsTUFBTSxHQUFHLGFBQWEsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3BELElBQUksS0FBSyxLQUFLLEVBQUUsRUFBRTtnQkFDZCxPQUFPLEtBQUssQ0FBQzthQUNoQjtZQUVELE9BQU8seUJBQXlCLENBQUM7UUFDckMsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBRVMsZUFBZSxDQUFDLGFBQXdDLEVBQUUsS0FBWSxFQUFFLGNBQThCO1FBQzVHLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDL0MsT0FBTztTQUNWO1FBRUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRTtZQUN2RCxPQUFPO1NBQ1Y7UUFFRCxNQUFNLFdBQVcsR0FBRyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2xGLE1BQU0sZUFBZSxHQUFHLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUM7UUFDeEYsTUFBTSxRQUFRLEdBQUcsY0FBYyxFQUFFLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRSxNQUFNLElBQUksRUFBRSxDQUFDO1FBRXBFLGFBQWEsQ0FBQyxnQkFBZ0IsR0FBRyxhQUFhLENBQUMsZ0JBQWdCLElBQUksRUFBRSxDQUFDO1FBQ3RFLE1BQU0sY0FBYyxHQUFHLEVBQUUsQ0FBQztRQUMxQixJQUFJLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLEVBQUU7WUFDdkMsYUFBYSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDM0MsY0FBYyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsR0FBRyxLQUFLLENBQUM7WUFDckMsQ0FBQyxDQUFDLENBQUE7U0FDTDtRQUVELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxhQUFhLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUNsRCxJQUFJLEtBQUssR0FBRyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRXBDLE1BQU0sV0FBVyxHQUFHLEVBQUUsQ0FBQztZQUN2QixXQUFXLENBQUMsZUFBZSxDQUFDLEdBQUcsS0FBSyxDQUFDO1lBQ3JDLFdBQVcsQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDO1lBRXRDLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUU7Z0JBQ3BDLGNBQWMsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxXQUFXLENBQUM7Z0JBQ2hELGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7YUFDcEQ7U0FDSjtJQUNMLENBQUM7SUFFUyxjQUFjLENBQUMsS0FBWTtRQUNqQyxJQUFJLFdBQVcsR0FBRyxJQUFJLENBQUM7UUFDdkIsTUFBTSxZQUFZLEdBQUcsS0FBSyxFQUFFLFVBQVUsRUFBRSxRQUFRLEVBQUUsWUFBWSxJQUFJLEVBQUUsQ0FBQztRQUVyRSxZQUFZLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQzFCLElBQUksTUFBTSxDQUFDLEtBQUssS0FBSyx5QkFBeUIsRUFBRTtnQkFDNUMsV0FBVyxHQUFHLE1BQU0sQ0FBQzthQUN4QjtRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxXQUFXLENBQUM7SUFDdkIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQixDQUFDLE1BQW1CLEVBQUUsU0FBaUI7UUFDcEUsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN2RCxPQUFPLENBQUMsQ0FBQyxhQUFhLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0IsQ0FBQyxXQUF3QixFQUFFLElBQVksRUFBRSxLQUFZO1FBRXhFLElBQUksQ0FBQyxXQUFXLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDakMsT0FBTztTQUNWO1FBRUQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUU7WUFDN0IsV0FBVyxDQUFDLGNBQWMsR0FBRyxFQUFFLENBQUM7U0FDbkM7UUFFRCxXQUFXLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQztRQUV6QyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7WUFDL0IsV0FBVyxDQUFDLFFBQVEsQ0FBQyxPQUFPLEdBQUcsRUFBRSxDQUFDO1NBQ3JDO1FBRUQsV0FBVyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUVwRCxJQUFJLFdBQVcsQ0FBQyxpQkFBaUIsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3BELFdBQVcsQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNyRTtJQUNMLENBQUM7bUZBak5RLGtCQUFrQjt1RUFBbEIsa0JBQWtCLFdBQWxCLGtCQUFrQixtQkFGZixNQUFNOztTQUVULGtCQUFrQjt1RkFBbEIsa0JBQWtCO2NBSDlCLFVBQVU7ZUFBQztnQkFDUixVQUFVLEVBQUUsTUFBTTthQUNyQiIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogU3VpdGVDUk0gaXMgYSBjdXN0b21lciByZWxhdGlvbnNoaXAgbWFuYWdlbWVudCBwcm9ncmFtIGRldmVsb3BlZCBieSBTYWxlc0FnaWxpdHkgTHRkLlxuICogQ29weXJpZ2h0IChDKSAyMDIxIFNhbGVzQWdpbGl0eSBMdGQuXG4gKlxuICogVGhpcyBwcm9ncmFtIGlzIGZyZWUgc29mdHdhcmU7IHlvdSBjYW4gcmVkaXN0cmlidXRlIGl0IGFuZC9vciBtb2RpZnkgaXQgdW5kZXJcbiAqIHRoZSB0ZXJtcyBvZiB0aGUgR05VIEFmZmVybyBHZW5lcmFsIFB1YmxpYyBMaWNlbnNlIHZlcnNpb24gMyBhcyBwdWJsaXNoZWQgYnkgdGhlXG4gKiBGcmVlIFNvZnR3YXJlIEZvdW5kYXRpb24gd2l0aCB0aGUgYWRkaXRpb24gb2YgdGhlIGZvbGxvd2luZyBwZXJtaXNzaW9uIGFkZGVkXG4gKiB0byBTZWN0aW9uIDE1IGFzIHBlcm1pdHRlZCBpbiBTZWN0aW9uIDcoYSk6IEZPUiBBTlkgUEFSVCBPRiBUSEUgQ09WRVJFRCBXT1JLXG4gKiBJTiBXSElDSCBUSEUgQ09QWVJJR0hUIElTIE9XTkVEIEJZIFNBTEVTQUdJTElUWSwgU0FMRVNBR0lMSVRZIERJU0NMQUlNUyBUSEVcbiAqIFdBUlJBTlRZIE9GIE5PTiBJTkZSSU5HRU1FTlQgT0YgVEhJUkQgUEFSVFkgUklHSFRTLlxuICpcbiAqIFRoaXMgcHJvZ3JhbSBpcyBkaXN0cmlidXRlZCBpbiB0aGUgaG9wZSB0aGF0IGl0IHdpbGwgYmUgdXNlZnVsLCBidXQgV0lUSE9VVFxuICogQU5ZIFdBUlJBTlRZOyB3aXRob3V0IGV2ZW4gdGhlIGltcGxpZWQgd2FycmFudHkgb2YgTUVSQ0hBTlRBQklMSVRZIG9yIEZJVE5FU1NcbiAqIEZPUiBBIFBBUlRJQ1VMQVIgUFVSUE9TRS4gU2VlIHRoZSBHTlUgQWZmZXJvIEdlbmVyYWwgUHVibGljIExpY2Vuc2UgZm9yIG1vcmVcbiAqIGRldGFpbHMuXG4gKlxuICogWW91IHNob3VsZCBoYXZlIHJlY2VpdmVkIGEgY29weSBvZiB0aGUgR05VIEFmZmVybyBHZW5lcmFsIFB1YmxpYyBMaWNlbnNlXG4gKiBhbG9uZyB3aXRoIHRoaXMgcHJvZ3JhbS4gIElmIG5vdCwgc2VlIDxodHRwOi8vd3d3LmdudS5vcmcvbGljZW5zZXMvPi5cbiAqXG4gKiBJbiBhY2NvcmRhbmNlIHdpdGggU2VjdGlvbiA3KGIpIG9mIHRoZSBHTlUgQWZmZXJvIEdlbmVyYWwgUHVibGljIExpY2Vuc2VcbiAqIHZlcnNpb24gMywgdGhlc2UgQXBwcm9wcmlhdGUgTGVnYWwgTm90aWNlcyBtdXN0IHJldGFpbiB0aGUgZGlzcGxheSBvZiB0aGVcbiAqIFwiU3VwZXJjaGFyZ2VkIGJ5IFN1aXRlQ1JNXCIgbG9nby4gSWYgdGhlIGRpc3BsYXkgb2YgdGhlIGxvZ29zIGlzIG5vdCByZWFzb25hYmx5XG4gKiBmZWFzaWJsZSBmb3IgdGVjaG5pY2FsIHJlYXNvbnMsIHRoZSBBcHByb3ByaWF0ZSBMZWdhbCBOb3RpY2VzIG11c3QgZGlzcGxheVxuICogdGhlIHdvcmRzIFwiU3VwZXJjaGFyZ2VkIGJ5IFN1aXRlQ1JNXCIuXG4gKi9cblxuaW1wb3J0IHtGaWVsZEJ1aWxkZXJ9IGZyb20gJy4vZmllbGQuYnVpbGRlcic7XG5pbXBvcnQge0luamVjdGFibGV9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xuaW1wb3J0IHtWYWxpZGF0aW9uTWFuYWdlcn0gZnJvbSAnLi4vdmFsaWRhdGlvbi92YWxpZGF0aW9uLm1hbmFnZXInO1xuaW1wb3J0IHtEYXRhVHlwZUZvcm1hdHRlcn0gZnJvbSAnLi4vLi4vZm9ybWF0dGVycy9kYXRhLXR5cGUuZm9ybWF0dGVyLnNlcnZpY2UnO1xuaW1wb3J0IHtTYXZlZEZpbHRlcn0gZnJvbSAnLi4vLi4vLi4vc3RvcmUvc2F2ZWQtZmlsdGVycy9zYXZlZC1maWx0ZXIubW9kZWwnO1xuaW1wb3J0IHtcbiAgICBkZWVwQ2xvbmUsXG4gICAgRmllbGQsXG4gICAgRmllbGREZWZpbml0aW9uLFxuICAgIE9wdGlvbixcbiAgICBTZWFyY2hDcml0ZXJpYSxcbiAgICBTZWFyY2hDcml0ZXJpYUZpZWxkRmlsdGVyLFxuICAgIFZpZXdGaWVsZERlZmluaXRpb25cbn0gZnJvbSAnY29tbW9uJztcbmltcG9ydCB7TGFuZ3VhZ2VTdG9yZX0gZnJvbSAnLi4vLi4vLi4vc3RvcmUvbGFuZ3VhZ2UvbGFuZ3VhZ2Uuc3RvcmUnO1xuaW1wb3J0IHtBc3luY1ZhbGlkYXRvckZuLCBWYWxpZGF0b3JGbn0gZnJvbSAnQGFuZ3VsYXIvZm9ybXMnO1xuaW1wb3J0IHtGaWVsZE9iamVjdFJlZ2lzdHJ5fSBmcm9tIFwiLi9maWVsZC1vYmplY3QtdHlwZS5yZWdpc3RyeVwiO1xuXG5ASW5qZWN0YWJsZSh7XG4gICAgcHJvdmlkZWRJbjogJ3Jvb3QnXG59KVxuZXhwb3J0IGNsYXNzIEZpbHRlckZpZWxkQnVpbGRlciBleHRlbmRzIEZpZWxkQnVpbGRlciB7XG5cbiAgICBjb25zdHJ1Y3RvcihcbiAgICAgICAgcHJvdGVjdGVkIHZhbGlkYXRpb25NYW5hZ2VyOiBWYWxpZGF0aW9uTWFuYWdlcixcbiAgICAgICAgcHJvdGVjdGVkIHR5cGVGb3JtYXR0ZXI6IERhdGFUeXBlRm9ybWF0dGVyLFxuICAgICAgICBwcm90ZWN0ZWQgZmllbGRSZWdpc3RyeTogRmllbGRPYmplY3RSZWdpc3RyeVxuICAgICkge1xuICAgICAgICBzdXBlcih2YWxpZGF0aW9uTWFuYWdlciwgdHlwZUZvcm1hdHRlciwgZmllbGRSZWdpc3RyeSk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQnVpbGQgZmlsdGVyIGZpZWxkXG4gICAgICpcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gc2F2ZWRGaWx0ZXIgU2F2ZWRGaWx0ZXJcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gdmlld0ZpZWxkIFZpZXdGaWVsZERlZmluaXRpb25cbiAgICAgKiBAcGFyYW0ge29iamVjdH0gbGFuZ3VhZ2UgTGFuZ3VhZ2VTdG9yZVxuICAgICAqIEByZXR1cm5zIHtvYmplY3R9IEZpZWxkXG4gICAgICovXG4gICAgcHVibGljIGJ1aWxkRmlsdGVyRmllbGQoc2F2ZWRGaWx0ZXI6IFNhdmVkRmlsdGVyLCB2aWV3RmllbGQ6IFZpZXdGaWVsZERlZmluaXRpb24sIGxhbmd1YWdlOiBMYW5ndWFnZVN0b3JlID0gbnVsbCk6IEZpZWxkIHtcblxuICAgICAgICBjb25zdCBkZWZpbml0aW9uID0gKHZpZXdGaWVsZCAmJiB2aWV3RmllbGQuZmllbGREZWZpbml0aW9uKSB8fCB7fSBhcyBGaWVsZERlZmluaXRpb247XG4gICAgICAgIGNvbnN0IHt2YWxpZGF0b3JzLCBhc3luY1ZhbGlkYXRvcnN9ID0gdGhpcy5nZXRGaWx0ZXJWYWxpZGF0b3JzKHNhdmVkRmlsdGVyLCB2aWV3RmllbGQpO1xuXG4gICAgICAgIGNvbnN0IGZpZWxkID0gdGhpcy5zZXR1cEZpZWxkKFxuICAgICAgICAgICAgc2F2ZWRGaWx0ZXIuc2VhcmNoTW9kdWxlLFxuICAgICAgICAgICAgdmlld0ZpZWxkLFxuICAgICAgICAgICAgbnVsbCxcbiAgICAgICAgICAgIG51bGwsXG4gICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgc2F2ZWRGaWx0ZXIsXG4gICAgICAgICAgICBkZWZpbml0aW9uLFxuICAgICAgICAgICAgdmFsaWRhdG9ycyxcbiAgICAgICAgICAgIGFzeW5jVmFsaWRhdG9ycyxcbiAgICAgICAgICAgIGxhbmd1YWdlXG4gICAgICAgICk7XG4gICAgICAgIGZpZWxkLmNyaXRlcmlhID0gdGhpcy5pbml0RmllbGRGaWx0ZXIoc2F2ZWRGaWx0ZXIuY3JpdGVyaWEsIHZpZXdGaWVsZCwgZmllbGQpO1xuICAgICAgICByZXR1cm4gZmllbGQ7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogR2V0IEZpbHRlciBWYWxpZGF0b3JzIGZvciBnaXZlbiBmaWVsZCBkZWZpbml0aW9uXG4gICAgICpcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gcmVjb3JkICBSZWNvcmRcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gdmlld0ZpZWxkIFZpZXdGaWVsZERlZmluaXRpb25cbiAgICAgKiBAcmV0dXJucyB7b2JqZWN0fSB2YWxpZGF0b3IgbWFwXG4gICAgICovXG4gICAgcHVibGljIGdldEZpbHRlclZhbGlkYXRvcnMoXG4gICAgICAgIHJlY29yZDogU2F2ZWRGaWx0ZXIsXG4gICAgICAgIHZpZXdGaWVsZDogVmlld0ZpZWxkRGVmaW5pdGlvblxuICAgICk6IHsgdmFsaWRhdG9yczogVmFsaWRhdG9yRm5bXTsgYXN5bmNWYWxpZGF0b3JzOiBBc3luY1ZhbGlkYXRvckZuW10gfSB7XG5cbiAgICAgICAgY29uc3QgdmFsaWRhdG9ycyA9IHRoaXMudmFsaWRhdGlvbk1hbmFnZXIuZ2V0RmlsdGVyVmFsaWRhdGlvbnMocmVjb3JkLnNlYXJjaE1vZHVsZSwgdmlld0ZpZWxkLCByZWNvcmQpO1xuICAgICAgICBjb25zdCBhc3luY1ZhbGlkYXRvcnM6IEFzeW5jVmFsaWRhdG9yRm5bXSA9IFtdO1xuXG4gICAgICAgIHJldHVybiB7dmFsaWRhdG9ycywgYXN5bmNWYWxpZGF0b3JzfTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBJbml0IGZpbHRlciBmaWVsZHNcbiAgICAgKlxuICAgICAqIEBwYXJhbSB7b2JqZWN0fSBzZWFyY2hDcml0ZXJpYSB0byB1c2VcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gdmlld0ZpZWxkIHRvIGluaXRcbiAgICAgKiBAcGFyYW0ge29iamVjdH0gZmllbGQgdG8gaW5pdFxuICAgICAqIEByZXR1cm5zIHtvYmplY3R9IFNlYXJjaENyaXRlcmlhRmllbGRGaWx0ZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgaW5pdEZpZWxkRmlsdGVyKHNlYXJjaENyaXRlcmlhOiBTZWFyY2hDcml0ZXJpYSwgdmlld0ZpZWxkOiBWaWV3RmllbGREZWZpbml0aW9uLCBmaWVsZDogRmllbGQpOiBTZWFyY2hDcml0ZXJpYUZpZWxkRmlsdGVyIHtcbiAgICAgICAgbGV0IGZpZWxkQ3JpdGVyaWE6IFNlYXJjaENyaXRlcmlhRmllbGRGaWx0ZXI7XG5cbiAgICAgICAgY29uc3QgZmllbGROYW1lID0gdmlld0ZpZWxkLm5hbWU7XG4gICAgICAgIGxldCBmaWVsZFR5cGUgPSB2aWV3RmllbGQudHlwZTtcbiAgICAgICAgbGV0IHJhbmdlU2VhcmNoID0gZmFsc2U7XG5cbiAgICAgICAgaWYgKGZpZWxkVHlwZSA9PT0gJ2NvbXBvc2l0ZScpIHtcbiAgICAgICAgICAgIGZpZWxkVHlwZSA9IGZpZWxkLmRlZmluaXRpb24udHlwZTtcbiAgICAgICAgICAgIHJhbmdlU2VhcmNoID0gdHJ1ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChzZWFyY2hDcml0ZXJpYS5maWx0ZXJzW2ZpZWxkTmFtZV0gJiYgc2VhcmNoQ3JpdGVyaWEuZmlsdGVyc1tmaWVsZE5hbWVdLmZpZWxkVHlwZSkge1xuICAgICAgICAgICAgZmllbGRDcml0ZXJpYSA9IGRlZXBDbG9uZShzZWFyY2hDcml0ZXJpYS5maWx0ZXJzW2ZpZWxkTmFtZV0pO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgZmllbGRDcml0ZXJpYSA9IHtcbiAgICAgICAgICAgICAgICBmaWVsZDogZmllbGROYW1lLFxuICAgICAgICAgICAgICAgIGZpZWxkVHlwZSxcbiAgICAgICAgICAgICAgICBvcGVyYXRvcjogJycsXG4gICAgICAgICAgICAgICAgdmFsdWVzOiBbXVxuICAgICAgICAgICAgfSBhcyBTZWFyY2hDcml0ZXJpYUZpZWxkRmlsdGVyO1xuICAgICAgICB9XG5cbiAgICAgICAgZmllbGRDcml0ZXJpYS5yYW5nZVNlYXJjaCA9IHJhbmdlU2VhcmNoO1xuXG4gICAgICAgIHRoaXMubWFwRW51bUVtcHR5T3B0aW9uKGZpZWxkQ3JpdGVyaWEsIGZpZWxkKTtcblxuICAgICAgICB0aGlzLm1hcFJlbGF0ZUZpZWxkcyhmaWVsZENyaXRlcmlhLCBmaWVsZCwgc2VhcmNoQ3JpdGVyaWEpO1xuXG4gICAgICAgIHJldHVybiBmaWVsZENyaXRlcmlhO1xuICAgIH1cblxuICAgIHByb3RlY3RlZCBtYXBFbnVtRW1wdHlPcHRpb24oZmllbGRDcml0ZXJpYTogU2VhcmNoQ3JpdGVyaWFGaWVsZEZpbHRlciwgZmllbGQ6IEZpZWxkKTogdm9pZCB7XG4gICAgICAgIGlmICghWydtdWx0aWVudW0nLCAnZW51bSddLmluY2x1ZGVzKGZpZWxkQ3JpdGVyaWEuZmllbGRUeXBlKSkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IGVtcHR5T3B0aW9uID0gdGhpcy5nZXRFbXB0eU9wdGlvbihmaWVsZCk7XG5cbiAgICAgICAgaWYgKCFlbXB0eU9wdGlvbikge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFmaWVsZENyaXRlcmlhLnZhbHVlcyB8fCAhZmllbGRDcml0ZXJpYS52YWx1ZXMubGVuZ3RoKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cblxuICAgICAgICBmaWVsZENyaXRlcmlhLnZhbHVlcyA9IGZpZWxkQ3JpdGVyaWEudmFsdWVzLm1hcCh2YWx1ZSA9PiB7XG4gICAgICAgICAgICBpZiAodmFsdWUgIT09ICcnKSB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHZhbHVlO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICByZXR1cm4gJ19fU3VpdGVDUk1FbXB0eVN0cmluZ19fJztcbiAgICAgICAgfSk7XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIG1hcFJlbGF0ZUZpZWxkcyhmaWVsZENyaXRlcmlhOiBTZWFyY2hDcml0ZXJpYUZpZWxkRmlsdGVyLCBmaWVsZDogRmllbGQsIHNlYXJjaENyaXRlcmlhOiBTZWFyY2hDcml0ZXJpYSk6IHZvaWQge1xuICAgICAgICBpZiAoIVsncmVsYXRlJ10uaW5jbHVkZXMoZmllbGRDcml0ZXJpYS5maWVsZFR5cGUpKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoIWZpZWxkQ3JpdGVyaWEudmFsdWVzIHx8ICFmaWVsZENyaXRlcmlhLnZhbHVlcy5sZW5ndGgpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IGlkRmllbGROYW1lID0gKGZpZWxkICYmIGZpZWxkLmRlZmluaXRpb24gJiYgZmllbGQuZGVmaW5pdGlvbi5pZF9uYW1lKSB8fCAnJztcbiAgICAgICAgY29uc3QgcmVsYXRlRmllbGROYW1lID0gKGZpZWxkICYmIGZpZWxkLmRlZmluaXRpb24gJiYgZmllbGQuZGVmaW5pdGlvbi5ybmFtZSkgfHwgJ25hbWUnO1xuICAgICAgICBjb25zdCBpZFZhbHVlcyA9IHNlYXJjaENyaXRlcmlhPy5maWx0ZXJzW2lkRmllbGROYW1lXT8udmFsdWVzID8/IFtdO1xuXG4gICAgICAgIGZpZWxkQ3JpdGVyaWEudmFsdWVPYmplY3RBcnJheSA9IGZpZWxkQ3JpdGVyaWEudmFsdWVPYmplY3RBcnJheSA/PyBbXTtcbiAgICAgICAgY29uc3QgcmVsYXRlRmllbGRNYXAgPSB7fTtcbiAgICAgICAgaWYgKGZpZWxkQ3JpdGVyaWEudmFsdWVPYmplY3RBcnJheS5sZW5ndGgpIHtcbiAgICAgICAgICAgIGZpZWxkQ3JpdGVyaWEudmFsdWVPYmplY3RBcnJheS5mb3JFYWNoKHZhbHVlID0+IHtcbiAgICAgICAgICAgICAgICByZWxhdGVGaWVsZE1hcFt2YWx1ZS5pZF0gPSB2YWx1ZTtcbiAgICAgICAgICAgIH0pXG4gICAgICAgIH1cblxuICAgICAgICBmb3IgKGxldCBpID0gMDsgaSA8IGZpZWxkQ3JpdGVyaWEudmFsdWVzLmxlbmd0aDsgaSsrKSB7XG4gICAgICAgICAgICBsZXQgdmFsdWUgPSBmaWVsZENyaXRlcmlhLnZhbHVlc1tpXTtcblxuICAgICAgICAgICAgY29uc3QgcmVsYXRlVmFsdWUgPSB7fTtcbiAgICAgICAgICAgIHJlbGF0ZVZhbHVlW3JlbGF0ZUZpZWxkTmFtZV0gPSB2YWx1ZTtcbiAgICAgICAgICAgIHJlbGF0ZVZhbHVlWydpZCddID0gaWRWYWx1ZXNbaV0gPz8gJyc7XG5cbiAgICAgICAgICAgIGlmICghcmVsYXRlRmllbGRNYXBbcmVsYXRlVmFsdWVbJ2lkJ11dKSB7XG4gICAgICAgICAgICAgICAgcmVsYXRlRmllbGRNYXBbcmVsYXRlVmFsdWVbJ2lkJ11dID0gcmVsYXRlVmFsdWU7XG4gICAgICAgICAgICAgICAgZmllbGRDcml0ZXJpYS52YWx1ZU9iamVjdEFycmF5LnB1c2gocmVsYXRlVmFsdWUpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIGdldEVtcHR5T3B0aW9uKGZpZWxkOiBGaWVsZCk6IE9wdGlvbiB7XG4gICAgICAgIGxldCBlbXB0eU9wdGlvbiA9IG51bGw7XG4gICAgICAgIGNvbnN0IGV4dHJhT3B0aW9ucyA9IGZpZWxkPy5kZWZpbml0aW9uPy5tZXRhZGF0YT8uZXh0cmFPcHRpb25zID8/IFtdO1xuXG4gICAgICAgIGV4dHJhT3B0aW9ucy5mb3JFYWNoKG9wdGlvbiA9PiB7XG4gICAgICAgICAgICBpZiAob3B0aW9uLnZhbHVlID09PSAnX19TdWl0ZUNSTUVtcHR5U3RyaW5nX18nKSB7XG4gICAgICAgICAgICAgICAgZW1wdHlPcHRpb24gPSBvcHRpb247XG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiBlbXB0eU9wdGlvbjtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBJcyBjcml0ZXJpYSBmaWVsZCBpbml0aWFsaXplZCBpbiByZWNvcmRcbiAgICAgKlxuICAgICAqIEBwYXJhbSB7b2JqZWN0fSByZWNvcmQgUmVjb3JkXG4gICAgICogQHBhcmFtIHtzdHJpbmd9IGZpZWxkTmFtZSBmaWVsZFxuICAgICAqIEByZXR1cm5zIHtib29sZWFufSBpc0luaXRpYWxpemVkXG4gICAgICovXG4gICAgcHVibGljIGlzQ3JpdGVyaWFGaWVsZEluaXRpYWxpemVkKHJlY29yZDogU2F2ZWRGaWx0ZXIsIGZpZWxkTmFtZTogc3RyaW5nKTogYm9vbGVhbiB7XG4gICAgICAgIGNvbnN0IGNyaXRlcmlhRmllbGQgPSByZWNvcmQuY3JpdGVyaWFGaWVsZHNbZmllbGROYW1lXTtcbiAgICAgICAgcmV0dXJuICEhY3JpdGVyaWFGaWVsZCAmJiAhY3JpdGVyaWFGaWVsZC52YXJkZWZCYXNlZDtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBBZGQgZmllbGQgdG8gU2F2ZWRGaWx0ZXJcbiAgICAgKlxuICAgICAqIEBwYXJhbSB7b2JqZWN0fSBzYXZlZEZpbHRlciBTYXZlZEZpbHRlclxuICAgICAqIEBwYXJhbSB7c3RyaW5nfSBuYW1lIHN0cmluZ1xuICAgICAqIEBwYXJhbSB7b2JqZWN0fSBmaWVsZCBGaWVsZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUb1NhdmVkRmlsdGVyKHNhdmVkRmlsdGVyOiBTYXZlZEZpbHRlciwgbmFtZTogc3RyaW5nLCBmaWVsZDogRmllbGQpOiB2b2lkIHtcblxuICAgICAgICBpZiAoIXNhdmVkRmlsdGVyIHx8ICFuYW1lIHx8ICFmaWVsZCkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFzYXZlZEZpbHRlci5jcml0ZXJpYUZpZWxkcykge1xuICAgICAgICAgICAgc2F2ZWRGaWx0ZXIuY3JpdGVyaWFGaWVsZHMgPSB7fTtcbiAgICAgICAgfVxuXG4gICAgICAgIHNhdmVkRmlsdGVyLmNyaXRlcmlhRmllbGRzW25hbWVdID0gZmllbGQ7XG5cbiAgICAgICAgaWYgKCFzYXZlZEZpbHRlci5jcml0ZXJpYS5maWx0ZXJzKSB7XG4gICAgICAgICAgICBzYXZlZEZpbHRlci5jcml0ZXJpYS5maWx0ZXJzID0ge307XG4gICAgICAgIH1cblxuICAgICAgICBzYXZlZEZpbHRlci5jcml0ZXJpYS5maWx0ZXJzW25hbWVdID0gZmllbGQuY3JpdGVyaWE7XG5cbiAgICAgICAgaWYgKHNhdmVkRmlsdGVyLmNyaXRlcmlhRm9ybUdyb3VwICYmIGZpZWxkLmZvcm1Db250cm9sKSB7XG4gICAgICAgICAgICBzYXZlZEZpbHRlci5jcml0ZXJpYUZvcm1Hcm91cC5hZGRDb250cm9sKG5hbWUsIGZpZWxkLmZvcm1Db250cm9sKTtcbiAgICAgICAgfVxuICAgIH1cbn1cbiJdfQ==