/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { BehaviorSubject, of } from 'rxjs';
import { catchError, map, shareReplay, tap } from 'rxjs/operators';
import { EntityGQL } from '../../services/api/graphql-api/api.entity.get';
import { deepClone, emptyObject } from 'common';
import { LanguageStore } from '../language/language.store';
import { SystemConfigStore } from '../system-config/system-config.store';
import { ThemeImagesStore } from '../theme-images/theme-images.store';
import { UserPreferenceStore } from '../user-preference/user-preference.store';
import { NavigationStore } from '../navigation/navigation.store';
import { MetadataStore } from '../metadata/metadata.store.service';
import { AdminMetadataStore } from '../admin-metadata/admin-metadata.store';
import { GlobalRecentlyViewedStore } from "../global-recently-viewed/global-recently-viewed.store";
import * as i0 from "@angular/core";
import * as i1 from "../../services/api/graphql-api/api.entity.get";
import * as i2 from "../metadata/metadata.store.service";
import * as i3 from "../language/language.store";
import * as i4 from "../theme-images/theme-images.store";
import * as i5 from "../system-config/system-config.store";
import * as i6 from "../user-preference/user-preference.store";
import * as i7 from "../navigation/navigation.store";
import * as i8 from "../admin-metadata/admin-metadata.store";
import * as i9 from "../global-recently-viewed/global-recently-viewed.store";
const initialState = {
    systemConfig: false,
    userPreferences: false,
    appStrings: false,
    appListStrings: false,
    modStrings: false,
    themeImages: false,
    navigation: false,
    moduleMetadata: false,
    adminMetadata: false,
    globalRecentlyViewed: false
};
let internalState = deepClone(initialState);
let cache$ = null;
class AppMetadataStore {
    constructor(recordGQL, metadata, language, themeImages, config, preferences, navigation, adminMetadataStore, globalRecentlyViewedStore) {
        this.recordGQL = recordGQL;
        this.metadata = metadata;
        this.language = language;
        this.themeImages = themeImages;
        this.config = config;
        this.preferences = preferences;
        this.navigation = navigation;
        this.adminMetadataStore = adminMetadataStore;
        this.globalRecentlyViewedStore = globalRecentlyViewedStore;
        this.store = new BehaviorSubject(internalState);
        this.state$ = this.store.asObservable();
        this.resourceName = 'appMetadata';
        this.fieldsMetadata = {
            fields: [
                'id',
                '_id',
            ]
        };
        this.types = [
            'systemConfig',
            'userPreferences',
            'language',
            'themeImages',
            'navigation',
            'moduleMetadata',
            'adminMetadata',
            'globalRecentlyViewed'
        ];
        this.metadata$ = this.state$;
    }
    /**
     * Clear state
     */
    clear() {
        cache$ = null;
        this.updateState(deepClone(initialState));
    }
    clearAuthBased() {
        this.clear();
    }
    get() {
        return internalState;
    }
    /**
     * Initial AppMetadata load if not cached and update state.
     *
     * @returns any data
     */
    load(module = 'login', types = [], useCache = true) {
        const notLoaded = this.getNotLoadedTypes();
        useCache = useCache && notLoaded.length < 1;
        if (!types || types.length < 1) {
            types = notLoaded;
            types.push('minimalModuleMetadata');
        }
        return this.getMetadata(module, types, useCache).pipe(tap((metadata) => {
            this.updateState(metadata);
        }));
    }
    /**
     * Initial AppMetadata load if not cached and update state.
     *
     * @returns any data
     */
    loadModuleMetadata(module = 'login', useCache = true) {
        let isLoaded = internalState?.moduleMetadata ?? false;
        useCache = useCache && isLoaded;
        return this.getMetadata(module, ['moduleMetadata'], useCache).pipe(tap((metadata) => {
            this.updateState(metadata);
        }));
    }
    /**
     * Get metadata cached Observable or call the backend
     *
     * @returns {object} Observable<AppMetadataFlags>
     */
    getMetadata(module = 'app', types = [], useCache = true) {
        if (!types || types.length < 1) {
            types = [...this.types];
        }
        if (cache$ == null || useCache !== true) {
            cache$ = this.fetch(module, types).pipe(shareReplay(1));
        }
        return cache$;
    }
    getNotLoadedTypes() {
        const types = [];
        if (!this.isNavigationLoaded()) {
            types.push('navigation');
        }
        if (!this.arePreferencesLoaded()) {
            types.push('userPreferences');
        }
        if (!this.areSystemConfigsLoaded()) {
            types.push('systemConfig');
        }
        if (!this.areAllLanguageStringsLoaded()) {
            types.push('language');
        }
        if (!this.areThemeImagesLoaded()) {
            types.push('themeImages');
        }
        if (!this.isAdminMetadataLoaded()) {
            types.push('adminMetadata');
        }
        if (!this.isGlobalRecentlyViewedLoaded()) {
            types.push('globalRecentlyViewed');
        }
        return types;
    }
    areAllLanguageStringsLoaded() {
        return this.language.areAllCached();
    }
    arePreferencesLoaded() {
        return this.preferences.isCached();
    }
    areSystemConfigsLoaded() {
        return this.config.isCached();
    }
    areThemeImagesLoaded() {
        return this.themeImages.isCached();
    }
    isNavigationLoaded() {
        return this.navigation.isCached();
    }
    isAdminMetadataLoaded() {
        return !!(internalState.adminMetadata ?? false);
    }
    isGlobalRecentlyViewedLoaded() {
        return !!(internalState.globalRecentlyViewed ?? false);
    }
    /**
     * Internal API
     */
    /**
     * Update the state
     *
     * @param {object} state to set
     */
    updateState(state) {
        this.store.next(internalState = state);
    }
    /**
     * Fetch the Metadata from the backend
     *
     * @returns {object} Observable<{}>
     */
    fetch(module, types = []) {
        const fieldsToRetrieve = {
            fields: [
                ...this.fieldsMetadata.fields,
                ...types
            ]
        };
        return this.recordGQL.fetch(this.resourceName, `/api/app-metadata/${module}`, fieldsToRetrieve)
            .pipe(catchError(() => {
            return of({});
        }), map(({ data }) => {
            const result = data?.appMetadata;
            const appMetadata = { ...internalState };
            if (emptyObject(result)) {
                return appMetadata;
            }
            this.setConfig(appMetadata, result);
            this.setPreferences(appMetadata, result);
            this.setThemeImages(appMetadata, result);
            this.setNavigation(appMetadata, result);
            this.setLanguages(appMetadata, result);
            this.setModuleMetadata(appMetadata, result);
            this.setAdminMetadata(appMetadata, result);
            this.setGlobalRecentlyViewed(appMetadata, result);
            return appMetadata;
        }));
    }
    setModuleMetadata(currentMetadata, appMetadata) {
        let moduleMetadata = appMetadata?.moduleMetadata ?? {};
        let metaKey = 'moduleMetadata';
        if (emptyObject(moduleMetadata)) {
            moduleMetadata = appMetadata?.minimalModuleMetadata ?? {};
            metaKey = 'minimalModuleMetadata';
        }
        if (!emptyObject(moduleMetadata)) {
            currentMetadata[metaKey] = true;
            Object.keys(moduleMetadata).forEach(module => {
                const meta = moduleMetadata[module] ?? {};
                if (!emptyObject(meta)) {
                    const parsedMeta = this.metadata.mapMetadata(module, moduleMetadata[module]);
                    if (this.metadata.getModule() !== module) {
                        this.metadata.setModuleMetadata(module, parsedMeta);
                    }
                    else if (!this.metadata.isCached(module)) {
                        this.metadata.set(module, parsedMeta);
                    }
                }
            });
        }
    }
    setLanguages(currentMetadata, appMetadata) {
        const lang = appMetadata?.language ?? {};
        if (!emptyObject(lang)) {
            const languageStrings = {};
            languageStrings.languageKey = lang.id ?? '';
            languageStrings.appStrings = lang?.appStrings?.items ?? {};
            languageStrings.appListStrings = lang?.appListStrings?.items ?? {};
            languageStrings.modStrings = lang?.modStrings?.items ?? {};
            currentMetadata.appStrings = !emptyObject(languageStrings.appStrings);
            currentMetadata.appListStrings = !emptyObject(languageStrings.appListStrings);
            currentMetadata.modStrings = !emptyObject(languageStrings.modStrings);
            this.language.set(languageStrings.languageKey, languageStrings);
        }
    }
    setNavigation(currentMetadata, appMetadata) {
        const navigation = appMetadata?.navigation ?? {};
        if (!emptyObject(navigation)) {
            currentMetadata.navigation = true;
            this.navigation.set(navigation);
        }
    }
    setThemeImages(currentMetadata, appMetadata) {
        const themeImages = appMetadata?.themeImages ?? {};
        const images = themeImages?.items ?? {};
        if (!emptyObject(themeImages) && !emptyObject(images)) {
            currentMetadata.themeImages = true;
            const theme = themeImages.id;
            this.themeImages.set(theme, images);
        }
    }
    setPreferences(currentMetadata, appMetadata) {
        const prefs = appMetadata?.userPreferences ?? {};
        if (!emptyObject(prefs)) {
            currentMetadata.userPreferences = true;
            const userPreferences = this.mapPreferences(prefs);
            this.preferences.set(userPreferences);
        }
    }
    setConfig(currentMetadata, appMetadata) {
        const configs = appMetadata?.systemConfig ?? {};
        if (!emptyObject(configs)) {
            currentMetadata.systemConfig = true;
            this.config.set(configs);
        }
    }
    setAdminMetadata(currentMetadata, appMetadata) {
        const adminMetadata = appMetadata?.adminMetadata ?? {};
        if (!emptyObject(adminMetadata)) {
            currentMetadata.adminMetadata = true;
            this.adminMetadataStore.set(adminMetadata);
        }
    }
    setGlobalRecentlyViewed(currentMetadata, appMetadata) {
        const globalRecentlyViewed = appMetadata?.globalRecentlyViewed ?? [];
        if (globalRecentlyViewed.length) {
            currentMetadata.globalRecentlyViewed = true;
            this.globalRecentlyViewedStore.set(globalRecentlyViewed);
        }
        else if (appMetadata?.globalRecentlyViewed) {
            this.globalRecentlyViewedStore.set(globalRecentlyViewed);
        }
    }
    mapPreferences(preferences) {
        const userPreferences = {};
        Object.keys(preferences).forEach((prefKey) => {
            if (!preferences[prefKey].items) {
                return;
            }
            Object.keys(preferences[prefKey].items).forEach(key => {
                userPreferences[key] = preferences[prefKey].items[key];
            });
        });
        return userPreferences;
    }
    static { this.ɵfac = function AppMetadataStore_Factory(t) { return new (t || AppMetadataStore)(i0.ɵɵinject(i1.EntityGQL), i0.ɵɵinject(i2.MetadataStore), i0.ɵɵinject(i3.LanguageStore), i0.ɵɵinject(i4.ThemeImagesStore), i0.ɵɵinject(i5.SystemConfigStore), i0.ɵɵinject(i6.UserPreferenceStore), i0.ɵɵinject(i7.NavigationStore), i0.ɵɵinject(i8.AdminMetadataStore), i0.ɵɵinject(i9.GlobalRecentlyViewedStore)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: AppMetadataStore, factory: AppMetadataStore.ɵfac, providedIn: 'root' }); }
}
export { AppMetadataStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(AppMetadataStore, [{
        type: Injectable,
        args: [{
                providedIn: 'root',
            }]
    }], function () { return [{ type: i1.EntityGQL }, { type: i2.MetadataStore }, { type: i3.LanguageStore }, { type: i4.ThemeImagesStore }, { type: i5.SystemConfigStore }, { type: i6.UserPreferenceStore }, { type: i7.NavigationStore }, { type: i8.AdminMetadataStore }, { type: i9.GlobalRecentlyViewedStore }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwLW1ldGFkYXRhLnN0b3JlLnNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi9jb3JlL2FwcC9jb3JlL3NyYy9saWIvc3RvcmUvYXBwLW1ldGFkYXRhL2FwcC1tZXRhZGF0YS5zdG9yZS5zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F3Qkc7QUFFSCxPQUFPLEVBQUMsVUFBVSxFQUFDLE1BQU0sZUFBZSxDQUFDO0FBQ3pDLE9BQU8sRUFBQyxlQUFlLEVBQWMsRUFBRSxFQUFDLE1BQU0sTUFBTSxDQUFDO0FBQ3JELE9BQU8sRUFBQyxVQUFVLEVBQUUsR0FBRyxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUMsTUFBTSxnQkFBZ0IsQ0FBQztBQUNqRSxPQUFPLEVBQUMsU0FBUyxFQUFDLE1BQU0sK0NBQStDLENBQUM7QUFDeEUsT0FBTyxFQUFDLFNBQVMsRUFBRSxXQUFXLEVBQUMsTUFBTSxRQUFRLENBQUM7QUFFOUMsT0FBTyxFQUFDLGFBQWEsRUFBa0IsTUFBTSw0QkFBNEIsQ0FBQztBQUMxRSxPQUFPLEVBQUMsaUJBQWlCLEVBQUMsTUFBTSxzQ0FBc0MsQ0FBQztBQUN2RSxPQUFPLEVBQUMsZ0JBQWdCLEVBQUMsTUFBTSxvQ0FBb0MsQ0FBQztBQUNwRSxPQUFPLEVBQW9CLG1CQUFtQixFQUFDLE1BQU0sMENBQTBDLENBQUM7QUFDaEcsT0FBTyxFQUFDLGVBQWUsRUFBQyxNQUFNLGdDQUFnQyxDQUFDO0FBQy9ELE9BQU8sRUFBd0IsYUFBYSxFQUFDLE1BQU0sb0NBQW9DLENBQUM7QUFFeEYsT0FBTyxFQUFDLGtCQUFrQixFQUFDLE1BQU0sd0NBQXdDLENBQUM7QUFFMUUsT0FBTyxFQUFDLHlCQUF5QixFQUFDLE1BQU0sd0RBQXdELENBQUM7Ozs7Ozs7Ozs7O0FBK0JqRyxNQUFNLFlBQVksR0FBcUI7SUFDbkMsWUFBWSxFQUFFLEtBQUs7SUFDbkIsZUFBZSxFQUFFLEtBQUs7SUFDdEIsVUFBVSxFQUFFLEtBQUs7SUFDakIsY0FBYyxFQUFFLEtBQUs7SUFDckIsVUFBVSxFQUFFLEtBQUs7SUFDakIsV0FBVyxFQUFFLEtBQUs7SUFDbEIsVUFBVSxFQUFFLEtBQUs7SUFDakIsY0FBYyxFQUFFLEtBQUs7SUFDckIsYUFBYSxFQUFFLEtBQUs7SUFDcEIsb0JBQW9CLEVBQUUsS0FBSztDQUM5QixDQUFDO0FBRUYsSUFBSSxhQUFhLEdBQXFCLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztBQUU5RCxJQUFJLE1BQU0sR0FBaUMsSUFBSSxDQUFDO0FBRWhELE1BR2EsZ0JBQWdCO0lBMkJ6QixZQUNjLFNBQW9CLEVBQ3BCLFFBQXVCLEVBQ3ZCLFFBQXVCLEVBQ3ZCLFdBQTZCLEVBQzdCLE1BQXlCLEVBQ3pCLFdBQWdDLEVBQ2hDLFVBQTJCLEVBQzNCLGtCQUFxQyxFQUNyQyx5QkFBb0Q7UUFScEQsY0FBUyxHQUFULFNBQVMsQ0FBVztRQUNwQixhQUFRLEdBQVIsUUFBUSxDQUFlO1FBQ3ZCLGFBQVEsR0FBUixRQUFRLENBQWU7UUFDdkIsZ0JBQVcsR0FBWCxXQUFXLENBQWtCO1FBQzdCLFdBQU0sR0FBTixNQUFNLENBQW1CO1FBQ3pCLGdCQUFXLEdBQVgsV0FBVyxDQUFxQjtRQUNoQyxlQUFVLEdBQVYsVUFBVSxDQUFpQjtRQUMzQix1QkFBa0IsR0FBbEIsa0JBQWtCLENBQW1CO1FBQ3JDLDhCQUF5QixHQUF6Qix5QkFBeUIsQ0FBMkI7UUE3QnhELFVBQUssR0FBRyxJQUFJLGVBQWUsQ0FBbUIsYUFBYSxDQUFDLENBQUM7UUFDN0QsV0FBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDbkMsaUJBQVksR0FBRyxhQUFhLENBQUM7UUFDN0IsbUJBQWMsR0FBRztZQUN2QixNQUFNLEVBQUU7Z0JBQ0osSUFBSTtnQkFDSixLQUFLO2FBQ1I7U0FDSixDQUFDO1FBQ1EsVUFBSyxHQUFHO1lBQ2QsY0FBYztZQUNkLGlCQUFpQjtZQUNqQixVQUFVO1lBQ1YsYUFBYTtZQUNiLFlBQVk7WUFDWixnQkFBZ0I7WUFDaEIsZUFBZTtZQUNmLHNCQUFzQjtTQUN6QixDQUFDO1FBYUUsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7T0FFRztJQUNJLEtBQUs7UUFDUixNQUFNLEdBQUcsSUFBSSxDQUFDO1FBQ2QsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRU0sY0FBYztRQUNqQixJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDakIsQ0FBQztJQUVNLEdBQUc7UUFDTixPQUFPLGFBQWEsQ0FBQztJQUN6QixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLElBQUksQ0FBQyxTQUFpQixPQUFPLEVBQUUsUUFBa0IsRUFBRSxFQUFFLFFBQVEsR0FBRyxJQUFJO1FBRXZFLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBQzNDLFFBQVEsR0FBRyxRQUFRLElBQUksU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFFNUMsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixLQUFLLEdBQUcsU0FBUyxDQUFDO1lBQ2xCLEtBQUssQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsQ0FBQztTQUN2QztRQUVELE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDLElBQUksQ0FDakQsR0FBRyxDQUFDLENBQUMsUUFBMEIsRUFBRSxFQUFFO1lBQy9CLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDL0IsQ0FBQyxDQUFDLENBQ0wsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0JBQWtCLENBQUMsU0FBaUIsT0FBTyxFQUFFLFFBQVEsR0FBRyxJQUFJO1FBQy9ELElBQUksUUFBUSxHQUFHLGFBQWEsRUFBRSxjQUFjLElBQUksS0FBSyxDQUFDO1FBRXRELFFBQVEsR0FBRyxRQUFRLElBQUksUUFBUSxDQUFDO1FBRWhDLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDLElBQUksQ0FDOUQsR0FBRyxDQUFDLENBQUMsUUFBMEIsRUFBRSxFQUFFO1lBQy9CLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDL0IsQ0FBQyxDQUFDLENBQ0wsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksV0FBVyxDQUFDLFNBQWlCLEtBQUssRUFBRSxRQUFrQixFQUFFLEVBQUUsUUFBUSxHQUFHLElBQUk7UUFFNUUsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixLQUFLLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMzQjtRQUVELElBQUksTUFBTSxJQUFJLElBQUksSUFBSSxRQUFRLEtBQUssSUFBSSxFQUFFO1lBQ3JDLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQ25DLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FDakIsQ0FBQztTQUNMO1FBRUQsT0FBTyxNQUFNLENBQUM7SUFDbEIsQ0FBQztJQUVTLGlCQUFpQjtRQUN2QixNQUFNLEtBQUssR0FBRyxFQUFFLENBQUM7UUFFakIsSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxFQUFFO1lBQzVCLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUE7U0FDM0I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFLEVBQUU7WUFDOUIsS0FBSyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFBO1NBQ2hDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxFQUFFO1lBQ2hDLEtBQUssQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUE7U0FDN0I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLDJCQUEyQixFQUFFLEVBQUU7WUFDckMsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQTtTQUN6QjtRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsRUFBRTtZQUM5QixLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFBO1NBQzVCO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxFQUFFO1lBQy9CLEtBQUssQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDL0I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEVBQUU7WUFDdEMsS0FBSyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1NBQ3RDO1FBRUQsT0FBTyxLQUFLLENBQUM7SUFDakIsQ0FBQztJQUVTLDJCQUEyQjtRQUNqQyxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLENBQUM7SUFDeEMsQ0FBQztJQUVTLG9CQUFvQjtRQUMxQixPQUFPLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDdkMsQ0FBQztJQUVTLHNCQUFzQjtRQUM1QixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDbEMsQ0FBQztJQUVTLG9CQUFvQjtRQUMxQixPQUFPLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDdkMsQ0FBQztJQUVTLGtCQUFrQjtRQUN4QixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDdEMsQ0FBQztJQUVTLHFCQUFxQjtRQUMzQixPQUFPLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxhQUFhLElBQUksS0FBSyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVTLDRCQUE0QjtRQUNsQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxvQkFBb0IsSUFBSSxLQUFLLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7O09BRUc7SUFFSDs7OztPQUlHO0lBQ08sV0FBVyxDQUFDLEtBQXVCO1FBQ3pDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNPLEtBQUssQ0FBQyxNQUFjLEVBQUUsUUFBa0IsRUFBRTtRQUNoRCxNQUFNLGdCQUFnQixHQUFHO1lBQ3JCLE1BQU0sRUFBRTtnQkFDSixHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTTtnQkFDN0IsR0FBRyxLQUFLO2FBQ1g7U0FDSixDQUFDO1FBRUYsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLHFCQUFxQixNQUFNLEVBQUUsRUFBRSxnQkFBZ0IsQ0FBQzthQUMxRixJQUFJLENBQ0QsVUFBVSxDQUFDLEdBQUcsRUFBRTtZQUNaLE9BQU8sRUFBRSxDQUFDLEVBQTRCLENBQUMsQ0FBQztRQUM1QyxDQUFDLENBQUMsRUFDRixHQUFHLENBQUMsQ0FBQyxFQUFDLElBQUksRUFBQyxFQUFFLEVBQUU7WUFFWCxNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsV0FBMEIsQ0FBQztZQUNoRCxNQUFNLFdBQVcsR0FBRyxFQUFDLEdBQUcsYUFBYSxFQUFxQixDQUFDO1lBQzNELElBQUksV0FBVyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUNyQixPQUFPLFdBQVcsQ0FBQzthQUN0QjtZQUVELElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQ3hDLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQ3ZDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDNUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsRUFBQyxNQUFNLENBQUMsQ0FBQztZQUMxQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsV0FBVyxFQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2pELE9BQU8sV0FBVyxDQUFDO1FBQ3ZCLENBQUMsQ0FBQyxDQUNMLENBQUM7SUFDVixDQUFDO0lBRVMsaUJBQWlCLENBQUMsZUFBaUMsRUFBRSxXQUF3QjtRQUNuRixJQUFJLGNBQWMsR0FBRyxXQUFXLEVBQUUsY0FBYyxJQUFJLEVBQUUsQ0FBQztRQUN2RCxJQUFJLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQztRQUUvQixJQUFJLFdBQVcsQ0FBQyxjQUFjLENBQUMsRUFBRTtZQUM3QixjQUFjLEdBQUcsV0FBVyxFQUFFLHFCQUFxQixJQUFJLEVBQUUsQ0FBQztZQUMxRCxPQUFPLEdBQUcsdUJBQXVCLENBQUM7U0FDckM7UUFFRCxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxFQUFFO1lBRTlCLGVBQWUsQ0FBQyxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUM7WUFFaEMsTUFBTSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3pDLE1BQU0sSUFBSSxHQUFHLGNBQWMsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFjLENBQUM7Z0JBQ3RELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUU7b0JBRXBCLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztvQkFFN0UsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxLQUFLLE1BQU0sRUFBRTt3QkFDdEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7cUJBQ3ZEO3lCQUFNLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTt3QkFDeEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDO3FCQUN6QztpQkFDSjtZQUNMLENBQUMsQ0FBQyxDQUFDO1NBQ047SUFDTCxDQUFDO0lBRVMsWUFBWSxDQUFDLGVBQWlDLEVBQUUsV0FBd0I7UUFDOUUsTUFBTSxJQUFJLEdBQUcsV0FBVyxFQUFFLFFBQVEsSUFBSSxFQUFFLENBQUM7UUFDekMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNwQixNQUFNLGVBQWUsR0FBRyxFQUFxQixDQUFDO1lBQzlDLGVBQWUsQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDNUMsZUFBZSxDQUFDLFVBQVUsR0FBRyxJQUFJLEVBQUUsVUFBVSxFQUFFLEtBQUssSUFBSSxFQUFFLENBQUM7WUFDM0QsZUFBZSxDQUFDLGNBQWMsR0FBRyxJQUFJLEVBQUUsY0FBYyxFQUFFLEtBQUssSUFBSSxFQUFFLENBQUM7WUFDbkUsZUFBZSxDQUFDLFVBQVUsR0FBRyxJQUFJLEVBQUUsVUFBVSxFQUFFLEtBQUssSUFBSSxFQUFFLENBQUM7WUFFM0QsZUFBZSxDQUFDLFVBQVUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDdEUsZUFBZSxDQUFDLGNBQWMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUFDLENBQUM7WUFDOUUsZUFBZSxDQUFDLFVBQVUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLENBQUM7WUFFdEUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRSxlQUFlLENBQUMsQ0FBQztTQUNuRTtJQUNMLENBQUM7SUFFUyxhQUFhLENBQUMsZUFBaUMsRUFBRSxXQUF3QjtRQUMvRSxNQUFNLFVBQVUsR0FBRyxXQUFXLEVBQUUsVUFBVSxJQUFJLEVBQUUsQ0FBQztRQUNqRCxJQUFJLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzFCLGVBQWUsQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDO1lBQ2xDLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ25DO0lBQ0wsQ0FBQztJQUVTLGNBQWMsQ0FBQyxlQUFpQyxFQUFFLFdBQXdCO1FBQ2hGLE1BQU0sV0FBVyxHQUFHLFdBQVcsRUFBRSxXQUFXLElBQUksRUFBRSxDQUFDO1FBQ25ELE1BQU0sTUFBTSxHQUFHLFdBQVcsRUFBRSxLQUFLLElBQUksRUFBRSxDQUFDO1FBQ3hDLElBQUksQ0FBQyxXQUFXLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDbkQsZUFBZSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUM7WUFDbkMsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7U0FDdkM7SUFDTCxDQUFDO0lBRVMsY0FBYyxDQUFDLGVBQWlDLEVBQUUsV0FBd0I7UUFDaEYsTUFBTSxLQUFLLEdBQUcsV0FBVyxFQUFFLGVBQWUsSUFBSSxFQUFFLENBQUM7UUFDakQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUNyQixlQUFlLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQztZQUN2QyxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ25ELElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ3pDO0lBQ0wsQ0FBQztJQUVTLFNBQVMsQ0FBQyxlQUFpQyxFQUFFLFdBQXdCO1FBQzNFLE1BQU0sT0FBTyxHQUFHLFdBQVcsRUFBRSxZQUFZLElBQUksRUFBRSxDQUFDO1FBQ2hELElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDdkIsZUFBZSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUM7WUFDcEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDNUI7SUFDTCxDQUFDO0lBRVMsZ0JBQWdCLENBQUMsZUFBaUMsRUFBRSxXQUF3QjtRQUNsRixNQUFNLGFBQWEsR0FBRyxXQUFXLEVBQUUsYUFBYSxJQUFJLEVBQUUsQ0FBQztRQUN2RCxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQzdCLGVBQWUsQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1lBQ3JDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDOUM7SUFDTCxDQUFDO0lBRVMsdUJBQXVCLENBQUMsZUFBaUMsRUFBRSxXQUF3QjtRQUN6RixNQUFNLG9CQUFvQixHQUFHLFdBQVcsRUFBRSxvQkFBb0IsSUFBSSxFQUFFLENBQUM7UUFDckUsSUFBSSxvQkFBb0IsQ0FBQyxNQUFNLEVBQUU7WUFDN0IsZUFBZSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQztZQUM1QyxJQUFJLENBQUMseUJBQXlCLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7U0FDNUQ7YUFBTSxJQUFHLFdBQVcsRUFBRSxvQkFBb0IsRUFBRTtZQUNyQyxJQUFJLENBQUMseUJBQXlCLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7U0FDaEU7SUFDTCxDQUFDO0lBRVMsY0FBYyxDQUFDLFdBQWdCO1FBQ3JDLE1BQU0sZUFBZSxHQUFzQixFQUFFLENBQUM7UUFDOUMsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUV6QyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssRUFBRTtnQkFDN0IsT0FBTzthQUNWO1lBRUQsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUNsRCxlQUFlLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUMzRCxDQUFDLENBQUMsQ0FBQztRQUNQLENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxlQUFlLENBQUM7SUFDM0IsQ0FBQztpRkF0VlEsZ0JBQWdCO3VFQUFoQixnQkFBZ0IsV0FBaEIsZ0JBQWdCLG1CQUZiLE1BQU07O1NBRVQsZ0JBQWdCO3VGQUFoQixnQkFBZ0I7Y0FINUIsVUFBVTtlQUFDO2dCQUNSLFVBQVUsRUFBRSxNQUFNO2FBQ3JCIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBTdWl0ZUNSTSBpcyBhIGN1c3RvbWVyIHJlbGF0aW9uc2hpcCBtYW5hZ2VtZW50IHByb2dyYW0gZGV2ZWxvcGVkIGJ5IFNhbGVzQWdpbGl0eSBMdGQuXG4gKiBDb3B5cmlnaHQgKEMpIDIwMjEgU2FsZXNBZ2lsaXR5IEx0ZC5cbiAqXG4gKiBUaGlzIHByb2dyYW0gaXMgZnJlZSBzb2Z0d2FyZTsgeW91IGNhbiByZWRpc3RyaWJ1dGUgaXQgYW5kL29yIG1vZGlmeSBpdCB1bmRlclxuICogdGhlIHRlcm1zIG9mIHRoZSBHTlUgQWZmZXJvIEdlbmVyYWwgUHVibGljIExpY2Vuc2UgdmVyc2lvbiAzIGFzIHB1Ymxpc2hlZCBieSB0aGVcbiAqIEZyZWUgU29mdHdhcmUgRm91bmRhdGlvbiB3aXRoIHRoZSBhZGRpdGlvbiBvZiB0aGUgZm9sbG93aW5nIHBlcm1pc3Npb24gYWRkZWRcbiAqIHRvIFNlY3Rpb24gMTUgYXMgcGVybWl0dGVkIGluIFNlY3Rpb24gNyhhKTogRk9SIEFOWSBQQVJUIE9GIFRIRSBDT1ZFUkVEIFdPUktcbiAqIElOIFdISUNIIFRIRSBDT1BZUklHSFQgSVMgT1dORUQgQlkgU0FMRVNBR0lMSVRZLCBTQUxFU0FHSUxJVFkgRElTQ0xBSU1TIFRIRVxuICogV0FSUkFOVFkgT0YgTk9OIElORlJJTkdFTUVOVCBPRiBUSElSRCBQQVJUWSBSSUdIVFMuXG4gKlxuICogVGhpcyBwcm9ncmFtIGlzIGRpc3RyaWJ1dGVkIGluIHRoZSBob3BlIHRoYXQgaXQgd2lsbCBiZSB1c2VmdWwsIGJ1dCBXSVRIT1VUXG4gKiBBTlkgV0FSUkFOVFk7IHdpdGhvdXQgZXZlbiB0aGUgaW1wbGllZCB3YXJyYW50eSBvZiBNRVJDSEFOVEFCSUxJVFkgb3IgRklUTkVTU1xuICogRk9SIEEgUEFSVElDVUxBUiBQVVJQT1NFLiBTZWUgdGhlIEdOVSBBZmZlcm8gR2VuZXJhbCBQdWJsaWMgTGljZW5zZSBmb3IgbW9yZVxuICogZGV0YWlscy5cbiAqXG4gKiBZb3Ugc2hvdWxkIGhhdmUgcmVjZWl2ZWQgYSBjb3B5IG9mIHRoZSBHTlUgQWZmZXJvIEdlbmVyYWwgUHVibGljIExpY2Vuc2VcbiAqIGFsb25nIHdpdGggdGhpcyBwcm9ncmFtLiAgSWYgbm90LCBzZWUgPGh0dHA6Ly93d3cuZ251Lm9yZy9saWNlbnNlcy8+LlxuICpcbiAqIEluIGFjY29yZGFuY2Ugd2l0aCBTZWN0aW9uIDcoYikgb2YgdGhlIEdOVSBBZmZlcm8gR2VuZXJhbCBQdWJsaWMgTGljZW5zZVxuICogdmVyc2lvbiAzLCB0aGVzZSBBcHByb3ByaWF0ZSBMZWdhbCBOb3RpY2VzIG11c3QgcmV0YWluIHRoZSBkaXNwbGF5IG9mIHRoZVxuICogXCJTdXBlcmNoYXJnZWQgYnkgU3VpdGVDUk1cIiBsb2dvLiBJZiB0aGUgZGlzcGxheSBvZiB0aGUgbG9nb3MgaXMgbm90IHJlYXNvbmFibHlcbiAqIGZlYXNpYmxlIGZvciB0ZWNobmljYWwgcmVhc29ucywgdGhlIEFwcHJvcHJpYXRlIExlZ2FsIE5vdGljZXMgbXVzdCBkaXNwbGF5XG4gKiB0aGUgd29yZHMgXCJTdXBlcmNoYXJnZWQgYnkgU3VpdGVDUk1cIi5cbiAqL1xuXG5pbXBvcnQge0luamVjdGFibGV9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xuaW1wb3J0IHtCZWhhdmlvclN1YmplY3QsIE9ic2VydmFibGUsIG9mfSBmcm9tICdyeGpzJztcbmltcG9ydCB7Y2F0Y2hFcnJvciwgbWFwLCBzaGFyZVJlcGxheSwgdGFwfSBmcm9tICdyeGpzL29wZXJhdG9ycyc7XG5pbXBvcnQge0VudGl0eUdRTH0gZnJvbSAnLi4vLi4vc2VydmljZXMvYXBpL2dyYXBocWwtYXBpL2FwaS5lbnRpdHkuZ2V0JztcbmltcG9ydCB7ZGVlcENsb25lLCBlbXB0eU9iamVjdH0gZnJvbSAnY29tbW9uJztcbmltcG9ydCB7U3RhdGVTdG9yZX0gZnJvbSAnLi4vc3RhdGUnO1xuaW1wb3J0IHtMYW5ndWFnZVN0b3JlLCBMYW5ndWFnZVN0cmluZ3N9IGZyb20gJy4uL2xhbmd1YWdlL2xhbmd1YWdlLnN0b3JlJztcbmltcG9ydCB7U3lzdGVtQ29uZmlnU3RvcmV9IGZyb20gJy4uL3N5c3RlbS1jb25maWcvc3lzdGVtLWNvbmZpZy5zdG9yZSc7XG5pbXBvcnQge1RoZW1lSW1hZ2VzU3RvcmV9IGZyb20gJy4uL3RoZW1lLWltYWdlcy90aGVtZS1pbWFnZXMuc3RvcmUnO1xuaW1wb3J0IHtVc2VyUHJlZmVyZW5jZU1hcCwgVXNlclByZWZlcmVuY2VTdG9yZX0gZnJvbSAnLi4vdXNlci1wcmVmZXJlbmNlL3VzZXItcHJlZmVyZW5jZS5zdG9yZSc7XG5pbXBvcnQge05hdmlnYXRpb25TdG9yZX0gZnJvbSAnLi4vbmF2aWdhdGlvbi9uYXZpZ2F0aW9uLnN0b3JlJztcbmltcG9ydCB7TWV0YWRhdGEsIE1ldGFkYXRhTWFwLCBNZXRhZGF0YVN0b3JlfSBmcm9tICcuLi9tZXRhZGF0YS9tZXRhZGF0YS5zdG9yZS5zZXJ2aWNlJztcbmltcG9ydCB7QXBvbGxvUXVlcnlSZXN1bHR9IGZyb20gJ0BhcG9sbG8vY2xpZW50L2NvcmUnO1xuaW1wb3J0IHtBZG1pbk1ldGFkYXRhU3RvcmV9IGZyb20gJy4uL2FkbWluLW1ldGFkYXRhL2FkbWluLW1ldGFkYXRhLnN0b3JlJztcbmltcG9ydCB7QWRtaW5NZXRhZGF0YX0gZnJvbSAnLi4vYWRtaW4tbWV0YWRhdGEvYWRtaW4tbWV0YWRhdGEubW9kZWwnO1xuaW1wb3J0IHtHbG9iYWxSZWNlbnRseVZpZXdlZFN0b3JlfSBmcm9tIFwiLi4vZ2xvYmFsLXJlY2VudGx5LXZpZXdlZC9nbG9iYWwtcmVjZW50bHktdmlld2VkLnN0b3JlXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBwTWV0YWRhdGEge1xuICAgIGxvYWRlZD86IGJvb2xlYW47XG4gICAgbW9kdWxlTWV0YWRhdGFMb2FkZWQ/OiBib29sZWFuO1xuICAgIHN5c3RlbUNvbmZpZz86IGFueTtcbiAgICB1c2VyUHJlZmVyZW5jZXM/OiBhbnk7XG4gICAgbGFuZ3VhZ2U/OiBhbnk7XG4gICAgdGhlbWVJbWFnZXM/OiBhbnk7XG4gICAgbmF2aWdhdGlvbj86IGFueTtcbiAgICBtb2R1bGVNZXRhZGF0YT86IE1ldGFkYXRhTWFwO1xuICAgIG1pbmltYWxNb2R1bGVNZXRhZGF0YT86IE1ldGFkYXRhTWFwO1xuICAgIGFkbWluTWV0YWRhdGE/OiBBZG1pbk1ldGFkYXRhO1xuICAgIGdsb2JhbFJlY2VudGx5Vmlld2VkPzogYW55O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEFwcE1ldGFkYXRhRmxhZ3Mge1xuICAgIHN5c3RlbUNvbmZpZz86IGJvb2xlYW47XG4gICAgdXNlclByZWZlcmVuY2VzPzogYm9vbGVhbjtcbiAgICBhcHBTdHJpbmdzPzogYm9vbGVhbjtcbiAgICBhcHBMaXN0U3RyaW5ncz86IGJvb2xlYW47XG4gICAgbW9kU3RyaW5ncz86IGJvb2xlYW47XG4gICAgdGhlbWVJbWFnZXM/OiBib29sZWFuO1xuICAgIG5hdmlnYXRpb24/OiBib29sZWFuO1xuICAgIG1vZHVsZU1ldGFkYXRhPzogYm9vbGVhbjtcbiAgICBtaW5pbWFsTW9kdWxlTWV0YWRhdGE/OiBib29sZWFuO1xuICAgIGFkbWluTWV0YWRhdGE/OiBib29sZWFuO1xuICAgIGdsb2JhbFJlY2VudGx5Vmlld2VkPzogYm9vbGVhbjtcbn1cblxuXG5jb25zdCBpbml0aWFsU3RhdGU6IEFwcE1ldGFkYXRhRmxhZ3MgPSB7XG4gICAgc3lzdGVtQ29uZmlnOiBmYWxzZSxcbiAgICB1c2VyUHJlZmVyZW5jZXM6IGZhbHNlLFxuICAgIGFwcFN0cmluZ3M6IGZhbHNlLFxuICAgIGFwcExpc3RTdHJpbmdzOiBmYWxzZSxcbiAgICBtb2RTdHJpbmdzOiBmYWxzZSxcbiAgICB0aGVtZUltYWdlczogZmFsc2UsXG4gICAgbmF2aWdhdGlvbjogZmFsc2UsXG4gICAgbW9kdWxlTWV0YWRhdGE6IGZhbHNlLFxuICAgIGFkbWluTWV0YWRhdGE6IGZhbHNlLFxuICAgIGdsb2JhbFJlY2VudGx5Vmlld2VkOiBmYWxzZVxufTtcblxubGV0IGludGVybmFsU3RhdGU6IEFwcE1ldGFkYXRhRmxhZ3MgPSBkZWVwQ2xvbmUoaW5pdGlhbFN0YXRlKTtcblxubGV0IGNhY2hlJDogT2JzZXJ2YWJsZTxBcHBNZXRhZGF0YUZsYWdzPiA9IG51bGw7XG5cbkBJbmplY3RhYmxlKHtcbiAgICBwcm92aWRlZEluOiAncm9vdCcsXG59KVxuZXhwb3J0IGNsYXNzIEFwcE1ldGFkYXRhU3RvcmUgaW1wbGVtZW50cyBTdGF0ZVN0b3JlIHtcblxuICAgIC8qKlxuICAgICAqIFB1YmxpYyBsb25nLWxpdmVkIG9ic2VydmFibGUgc3RyZWFtc1xuICAgICAqL1xuICAgIG1ldGFkYXRhJDogT2JzZXJ2YWJsZTxBcHBNZXRhZGF0YUZsYWdzPjtcblxuICAgIHByb3RlY3RlZCBzdG9yZSA9IG5ldyBCZWhhdmlvclN1YmplY3Q8QXBwTWV0YWRhdGFGbGFncz4oaW50ZXJuYWxTdGF0ZSk7XG4gICAgcHJvdGVjdGVkIHN0YXRlJCA9IHRoaXMuc3RvcmUuYXNPYnNlcnZhYmxlKCk7XG4gICAgcHJvdGVjdGVkIHJlc291cmNlTmFtZSA9ICdhcHBNZXRhZGF0YSc7XG4gICAgcHJvdGVjdGVkIGZpZWxkc01ldGFkYXRhID0ge1xuICAgICAgICBmaWVsZHM6IFtcbiAgICAgICAgICAgICdpZCcsXG4gICAgICAgICAgICAnX2lkJyxcbiAgICAgICAgXVxuICAgIH07XG4gICAgcHJvdGVjdGVkIHR5cGVzID0gW1xuICAgICAgICAnc3lzdGVtQ29uZmlnJyxcbiAgICAgICAgJ3VzZXJQcmVmZXJlbmNlcycsXG4gICAgICAgICdsYW5ndWFnZScsXG4gICAgICAgICd0aGVtZUltYWdlcycsXG4gICAgICAgICduYXZpZ2F0aW9uJyxcbiAgICAgICAgJ21vZHVsZU1ldGFkYXRhJyxcbiAgICAgICAgJ2FkbWluTWV0YWRhdGEnLFxuICAgICAgICAnZ2xvYmFsUmVjZW50bHlWaWV3ZWQnXG4gICAgXTtcblxuICAgIGNvbnN0cnVjdG9yKFxuICAgICAgICBwcm90ZWN0ZWQgcmVjb3JkR1FMOiBFbnRpdHlHUUwsXG4gICAgICAgIHByb3RlY3RlZCBtZXRhZGF0YTogTWV0YWRhdGFTdG9yZSxcbiAgICAgICAgcHJvdGVjdGVkIGxhbmd1YWdlOiBMYW5ndWFnZVN0b3JlLFxuICAgICAgICBwcm90ZWN0ZWQgdGhlbWVJbWFnZXM6IFRoZW1lSW1hZ2VzU3RvcmUsXG4gICAgICAgIHByb3RlY3RlZCBjb25maWc6IFN5c3RlbUNvbmZpZ1N0b3JlLFxuICAgICAgICBwcm90ZWN0ZWQgcHJlZmVyZW5jZXM6IFVzZXJQcmVmZXJlbmNlU3RvcmUsXG4gICAgICAgIHByb3RlY3RlZCBuYXZpZ2F0aW9uOiBOYXZpZ2F0aW9uU3RvcmUsXG4gICAgICAgIHByb3RlY3RlZCBhZG1pbk1ldGFkYXRhU3RvcmU6QWRtaW5NZXRhZGF0YVN0b3JlLFxuICAgICAgICBwcm90ZWN0ZWQgZ2xvYmFsUmVjZW50bHlWaWV3ZWRTdG9yZTogR2xvYmFsUmVjZW50bHlWaWV3ZWRTdG9yZVxuICAgICkge1xuICAgICAgICB0aGlzLm1ldGFkYXRhJCA9IHRoaXMuc3RhdGUkO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIENsZWFyIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIGNsZWFyKCk6IHZvaWQge1xuICAgICAgICBjYWNoZSQgPSBudWxsO1xuICAgICAgICB0aGlzLnVwZGF0ZVN0YXRlKGRlZXBDbG9uZShpbml0aWFsU3RhdGUpKTtcbiAgICB9XG5cbiAgICBwdWJsaWMgY2xlYXJBdXRoQmFzZWQoKTogdm9pZCB7XG4gICAgICAgIHRoaXMuY2xlYXIoKTtcbiAgICB9XG5cbiAgICBwdWJsaWMgZ2V0KCk6IEFwcE1ldGFkYXRhRmxhZ3Mge1xuICAgICAgICByZXR1cm4gaW50ZXJuYWxTdGF0ZTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBJbml0aWFsIEFwcE1ldGFkYXRhIGxvYWQgaWYgbm90IGNhY2hlZCBhbmQgdXBkYXRlIHN0YXRlLlxuICAgICAqXG4gICAgICogQHJldHVybnMgYW55IGRhdGFcbiAgICAgKi9cbiAgICBwdWJsaWMgbG9hZChtb2R1bGU6IHN0cmluZyA9ICdsb2dpbicsIHR5cGVzOiBzdHJpbmdbXSA9IFtdLCB1c2VDYWNoZSA9IHRydWUpOiBPYnNlcnZhYmxlPEFwcE1ldGFkYXRhRmxhZ3M+IHtcblxuICAgICAgICBjb25zdCBub3RMb2FkZWQgPSB0aGlzLmdldE5vdExvYWRlZFR5cGVzKCk7XG4gICAgICAgIHVzZUNhY2hlID0gdXNlQ2FjaGUgJiYgbm90TG9hZGVkLmxlbmd0aCA8IDE7XG5cbiAgICAgICAgaWYgKCF0eXBlcyB8fCB0eXBlcy5sZW5ndGggPCAxKSB7XG4gICAgICAgICAgICB0eXBlcyA9IG5vdExvYWRlZDtcbiAgICAgICAgICAgIHR5cGVzLnB1c2goJ21pbmltYWxNb2R1bGVNZXRhZGF0YScpO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIHRoaXMuZ2V0TWV0YWRhdGEobW9kdWxlLCB0eXBlcywgdXNlQ2FjaGUpLnBpcGUoXG4gICAgICAgICAgICB0YXAoKG1ldGFkYXRhOiBBcHBNZXRhZGF0YUZsYWdzKSA9PiB7XG4gICAgICAgICAgICAgICAgdGhpcy51cGRhdGVTdGF0ZShtZXRhZGF0YSk7XG4gICAgICAgICAgICB9KVxuICAgICAgICApO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEluaXRpYWwgQXBwTWV0YWRhdGEgbG9hZCBpZiBub3QgY2FjaGVkIGFuZCB1cGRhdGUgc3RhdGUuXG4gICAgICpcbiAgICAgKiBAcmV0dXJucyBhbnkgZGF0YVxuICAgICAqL1xuICAgIHB1YmxpYyBsb2FkTW9kdWxlTWV0YWRhdGEobW9kdWxlOiBzdHJpbmcgPSAnbG9naW4nLCB1c2VDYWNoZSA9IHRydWUpOiBPYnNlcnZhYmxlPEFwcE1ldGFkYXRhRmxhZ3M+IHtcbiAgICAgICAgbGV0IGlzTG9hZGVkID0gaW50ZXJuYWxTdGF0ZT8ubW9kdWxlTWV0YWRhdGEgPz8gZmFsc2U7XG5cbiAgICAgICAgdXNlQ2FjaGUgPSB1c2VDYWNoZSAmJiBpc0xvYWRlZDtcblxuICAgICAgICByZXR1cm4gdGhpcy5nZXRNZXRhZGF0YShtb2R1bGUsIFsnbW9kdWxlTWV0YWRhdGEnXSwgdXNlQ2FjaGUpLnBpcGUoXG4gICAgICAgICAgICB0YXAoKG1ldGFkYXRhOiBBcHBNZXRhZGF0YUZsYWdzKSA9PiB7XG4gICAgICAgICAgICAgICAgdGhpcy51cGRhdGVTdGF0ZShtZXRhZGF0YSk7XG4gICAgICAgICAgICB9KVxuICAgICAgICApO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEdldCBtZXRhZGF0YSBjYWNoZWQgT2JzZXJ2YWJsZSBvciBjYWxsIHRoZSBiYWNrZW5kXG4gICAgICpcbiAgICAgKiBAcmV0dXJucyB7b2JqZWN0fSBPYnNlcnZhYmxlPEFwcE1ldGFkYXRhRmxhZ3M+XG4gICAgICovXG4gICAgcHVibGljIGdldE1ldGFkYXRhKG1vZHVsZTogc3RyaW5nID0gJ2FwcCcsIHR5cGVzOiBzdHJpbmdbXSA9IFtdLCB1c2VDYWNoZSA9IHRydWUpOiBPYnNlcnZhYmxlPEFwcE1ldGFkYXRhRmxhZ3M+IHtcblxuICAgICAgICBpZiAoIXR5cGVzIHx8IHR5cGVzLmxlbmd0aCA8IDEpIHtcbiAgICAgICAgICAgIHR5cGVzID0gWy4uLnRoaXMudHlwZXNdO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGNhY2hlJCA9PSBudWxsIHx8IHVzZUNhY2hlICE9PSB0cnVlKSB7XG4gICAgICAgICAgICBjYWNoZSQgPSB0aGlzLmZldGNoKG1vZHVsZSwgdHlwZXMpLnBpcGUoXG4gICAgICAgICAgICAgICAgc2hhcmVSZXBsYXkoMSlcbiAgICAgICAgICAgICk7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gY2FjaGUkO1xuICAgIH1cblxuICAgIHByb3RlY3RlZCBnZXROb3RMb2FkZWRUeXBlcygpOiBzdHJpbmdbXSB7XG4gICAgICAgIGNvbnN0IHR5cGVzID0gW107XG5cbiAgICAgICAgaWYgKCF0aGlzLmlzTmF2aWdhdGlvbkxvYWRlZCgpKSB7XG4gICAgICAgICAgICB0eXBlcy5wdXNoKCduYXZpZ2F0aW9uJylcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghdGhpcy5hcmVQcmVmZXJlbmNlc0xvYWRlZCgpKSB7XG4gICAgICAgICAgICB0eXBlcy5wdXNoKCd1c2VyUHJlZmVyZW5jZXMnKVxuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCF0aGlzLmFyZVN5c3RlbUNvbmZpZ3NMb2FkZWQoKSkge1xuICAgICAgICAgICAgdHlwZXMucHVzaCgnc3lzdGVtQ29uZmlnJylcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghdGhpcy5hcmVBbGxMYW5ndWFnZVN0cmluZ3NMb2FkZWQoKSkge1xuICAgICAgICAgICAgdHlwZXMucHVzaCgnbGFuZ3VhZ2UnKVxuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCF0aGlzLmFyZVRoZW1lSW1hZ2VzTG9hZGVkKCkpIHtcbiAgICAgICAgICAgIHR5cGVzLnB1c2goJ3RoZW1lSW1hZ2VzJylcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghdGhpcy5pc0FkbWluTWV0YWRhdGFMb2FkZWQoKSkge1xuICAgICAgICAgICAgdHlwZXMucHVzaCgnYWRtaW5NZXRhZGF0YScpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCF0aGlzLmlzR2xvYmFsUmVjZW50bHlWaWV3ZWRMb2FkZWQoKSkge1xuICAgICAgICAgICAgdHlwZXMucHVzaCgnZ2xvYmFsUmVjZW50bHlWaWV3ZWQnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiB0eXBlcztcbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgYXJlQWxsTGFuZ3VhZ2VTdHJpbmdzTG9hZGVkKCk6IGJvb2xlYW4ge1xuICAgICAgICByZXR1cm4gdGhpcy5sYW5ndWFnZS5hcmVBbGxDYWNoZWQoKTtcbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgYXJlUHJlZmVyZW5jZXNMb2FkZWQoKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0aGlzLnByZWZlcmVuY2VzLmlzQ2FjaGVkKCk7XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIGFyZVN5c3RlbUNvbmZpZ3NMb2FkZWQoKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0aGlzLmNvbmZpZy5pc0NhY2hlZCgpO1xuICAgIH1cblxuICAgIHByb3RlY3RlZCBhcmVUaGVtZUltYWdlc0xvYWRlZCgpOiBib29sZWFuIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGhlbWVJbWFnZXMuaXNDYWNoZWQoKTtcbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgaXNOYXZpZ2F0aW9uTG9hZGVkKCk6IGJvb2xlYW4ge1xuICAgICAgICByZXR1cm4gdGhpcy5uYXZpZ2F0aW9uLmlzQ2FjaGVkKCk7XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIGlzQWRtaW5NZXRhZGF0YUxvYWRlZCgpOiBib29sZWFuIHtcbiAgICAgICAgcmV0dXJuICEhKGludGVybmFsU3RhdGUuYWRtaW5NZXRhZGF0YSA/PyBmYWxzZSk7XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIGlzR2xvYmFsUmVjZW50bHlWaWV3ZWRMb2FkZWQoKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiAhIShpbnRlcm5hbFN0YXRlLmdsb2JhbFJlY2VudGx5Vmlld2VkID8/IGZhbHNlKTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBJbnRlcm5hbCBBUElcbiAgICAgKi9cblxuICAgIC8qKlxuICAgICAqIFVwZGF0ZSB0aGUgc3RhdGVcbiAgICAgKlxuICAgICAqIEBwYXJhbSB7b2JqZWN0fSBzdGF0ZSB0byBzZXRcbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgdXBkYXRlU3RhdGUoc3RhdGU6IEFwcE1ldGFkYXRhRmxhZ3MpOiB2b2lkIHtcbiAgICAgICAgdGhpcy5zdG9yZS5uZXh0KGludGVybmFsU3RhdGUgPSBzdGF0ZSk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogRmV0Y2ggdGhlIE1ldGFkYXRhIGZyb20gdGhlIGJhY2tlbmRcbiAgICAgKlxuICAgICAqIEByZXR1cm5zIHtvYmplY3R9IE9ic2VydmFibGU8e30+XG4gICAgICovXG4gICAgcHJvdGVjdGVkIGZldGNoKG1vZHVsZTogc3RyaW5nLCB0eXBlczogc3RyaW5nW10gPSBbXSk6IE9ic2VydmFibGU8QXBwTWV0YWRhdGFGbGFncz4ge1xuICAgICAgICBjb25zdCBmaWVsZHNUb1JldHJpZXZlID0ge1xuICAgICAgICAgICAgZmllbGRzOiBbXG4gICAgICAgICAgICAgICAgLi4udGhpcy5maWVsZHNNZXRhZGF0YS5maWVsZHMsXG4gICAgICAgICAgICAgICAgLi4udHlwZXNcbiAgICAgICAgICAgIF1cbiAgICAgICAgfTtcblxuICAgICAgICByZXR1cm4gdGhpcy5yZWNvcmRHUUwuZmV0Y2godGhpcy5yZXNvdXJjZU5hbWUsIGAvYXBpL2FwcC1tZXRhZGF0YS8ke21vZHVsZX1gLCBmaWVsZHNUb1JldHJpZXZlKVxuICAgICAgICAgICAgLnBpcGUoXG4gICAgICAgICAgICAgICAgY2F0Y2hFcnJvcigoKSA9PiB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBvZih7fSBhcyBBcG9sbG9RdWVyeVJlc3VsdDxhbnk+KTtcbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICBtYXAoKHtkYXRhfSkgPT4ge1xuXG4gICAgICAgICAgICAgICAgICAgIGNvbnN0IHJlc3VsdCA9IGRhdGE/LmFwcE1ldGFkYXRhIGFzIEFwcE1ldGFkYXRhO1xuICAgICAgICAgICAgICAgICAgICBjb25zdCBhcHBNZXRhZGF0YSA9IHsuLi5pbnRlcm5hbFN0YXRlfSBhcyBBcHBNZXRhZGF0YUZsYWdzO1xuICAgICAgICAgICAgICAgICAgICBpZiAoZW1wdHlPYmplY3QocmVzdWx0KSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIGFwcE1ldGFkYXRhO1xuICAgICAgICAgICAgICAgICAgICB9XG5cbiAgICAgICAgICAgICAgICAgICAgdGhpcy5zZXRDb25maWcoYXBwTWV0YWRhdGEsIHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuc2V0UHJlZmVyZW5jZXMoYXBwTWV0YWRhdGEsIHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuc2V0VGhlbWVJbWFnZXMoYXBwTWV0YWRhdGEsIHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuc2V0TmF2aWdhdGlvbihhcHBNZXRhZGF0YSwgcmVzdWx0KTtcbiAgICAgICAgICAgICAgICAgICAgdGhpcy5zZXRMYW5ndWFnZXMoYXBwTWV0YWRhdGEsIHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuc2V0TW9kdWxlTWV0YWRhdGEoYXBwTWV0YWRhdGEsIHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuc2V0QWRtaW5NZXRhZGF0YShhcHBNZXRhZGF0YSxyZXN1bHQpO1xuICAgICAgICAgICAgICAgICAgICB0aGlzLnNldEdsb2JhbFJlY2VudGx5Vmlld2VkKGFwcE1ldGFkYXRhLHJlc3VsdCk7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBhcHBNZXRhZGF0YTtcbiAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgKTtcbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgc2V0TW9kdWxlTWV0YWRhdGEoY3VycmVudE1ldGFkYXRhOiBBcHBNZXRhZGF0YUZsYWdzLCBhcHBNZXRhZGF0YTogQXBwTWV0YWRhdGEpIHtcbiAgICAgICAgbGV0IG1vZHVsZU1ldGFkYXRhID0gYXBwTWV0YWRhdGE/Lm1vZHVsZU1ldGFkYXRhID8/IHt9O1xuICAgICAgICBsZXQgbWV0YUtleSA9ICdtb2R1bGVNZXRhZGF0YSc7XG5cbiAgICAgICAgaWYgKGVtcHR5T2JqZWN0KG1vZHVsZU1ldGFkYXRhKSkge1xuICAgICAgICAgICAgbW9kdWxlTWV0YWRhdGEgPSBhcHBNZXRhZGF0YT8ubWluaW1hbE1vZHVsZU1ldGFkYXRhID8/IHt9O1xuICAgICAgICAgICAgbWV0YUtleSA9ICdtaW5pbWFsTW9kdWxlTWV0YWRhdGEnO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFlbXB0eU9iamVjdChtb2R1bGVNZXRhZGF0YSkpIHtcblxuICAgICAgICAgICAgY3VycmVudE1ldGFkYXRhW21ldGFLZXldID0gdHJ1ZTtcblxuICAgICAgICAgICAgT2JqZWN0LmtleXMobW9kdWxlTWV0YWRhdGEpLmZvckVhY2gobW9kdWxlID0+IHtcbiAgICAgICAgICAgICAgICBjb25zdCBtZXRhID0gbW9kdWxlTWV0YWRhdGFbbW9kdWxlXSA/PyB7fSBhcyBNZXRhZGF0YTtcbiAgICAgICAgICAgICAgICBpZiAoIWVtcHR5T2JqZWN0KG1ldGEpKSB7XG5cbiAgICAgICAgICAgICAgICAgICAgY29uc3QgcGFyc2VkTWV0YSA9IHRoaXMubWV0YWRhdGEubWFwTWV0YWRhdGEobW9kdWxlLCBtb2R1bGVNZXRhZGF0YVttb2R1bGVdKTtcblxuICAgICAgICAgICAgICAgICAgICBpZiAodGhpcy5tZXRhZGF0YS5nZXRNb2R1bGUoKSAhPT0gbW9kdWxlKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0aGlzLm1ldGFkYXRhLnNldE1vZHVsZU1ldGFkYXRhKG1vZHVsZSwgcGFyc2VkTWV0YSk7XG4gICAgICAgICAgICAgICAgICAgIH0gZWxzZSBpZiAoIXRoaXMubWV0YWRhdGEuaXNDYWNoZWQobW9kdWxlKSkge1xuICAgICAgICAgICAgICAgICAgICAgICAgdGhpcy5tZXRhZGF0YS5zZXQobW9kdWxlLCBwYXJzZWRNZXRhKTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIHNldExhbmd1YWdlcyhjdXJyZW50TWV0YWRhdGE6IEFwcE1ldGFkYXRhRmxhZ3MsIGFwcE1ldGFkYXRhOiBBcHBNZXRhZGF0YSkge1xuICAgICAgICBjb25zdCBsYW5nID0gYXBwTWV0YWRhdGE/Lmxhbmd1YWdlID8/IHt9O1xuICAgICAgICBpZiAoIWVtcHR5T2JqZWN0KGxhbmcpKSB7XG4gICAgICAgICAgICBjb25zdCBsYW5ndWFnZVN0cmluZ3MgPSB7fSBhcyBMYW5ndWFnZVN0cmluZ3M7XG4gICAgICAgICAgICBsYW5ndWFnZVN0cmluZ3MubGFuZ3VhZ2VLZXkgPSBsYW5nLmlkID8/ICcnO1xuICAgICAgICAgICAgbGFuZ3VhZ2VTdHJpbmdzLmFwcFN0cmluZ3MgPSBsYW5nPy5hcHBTdHJpbmdzPy5pdGVtcyA/PyB7fTtcbiAgICAgICAgICAgIGxhbmd1YWdlU3RyaW5ncy5hcHBMaXN0U3RyaW5ncyA9IGxhbmc/LmFwcExpc3RTdHJpbmdzPy5pdGVtcyA/PyB7fTtcbiAgICAgICAgICAgIGxhbmd1YWdlU3RyaW5ncy5tb2RTdHJpbmdzID0gbGFuZz8ubW9kU3RyaW5ncz8uaXRlbXMgPz8ge307XG5cbiAgICAgICAgICAgIGN1cnJlbnRNZXRhZGF0YS5hcHBTdHJpbmdzID0gIWVtcHR5T2JqZWN0KGxhbmd1YWdlU3RyaW5ncy5hcHBTdHJpbmdzKTtcbiAgICAgICAgICAgIGN1cnJlbnRNZXRhZGF0YS5hcHBMaXN0U3RyaW5ncyA9ICFlbXB0eU9iamVjdChsYW5ndWFnZVN0cmluZ3MuYXBwTGlzdFN0cmluZ3MpO1xuICAgICAgICAgICAgY3VycmVudE1ldGFkYXRhLm1vZFN0cmluZ3MgPSAhZW1wdHlPYmplY3QobGFuZ3VhZ2VTdHJpbmdzLm1vZFN0cmluZ3MpO1xuXG4gICAgICAgICAgICB0aGlzLmxhbmd1YWdlLnNldChsYW5ndWFnZVN0cmluZ3MubGFuZ3VhZ2VLZXksIGxhbmd1YWdlU3RyaW5ncyk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgc2V0TmF2aWdhdGlvbihjdXJyZW50TWV0YWRhdGE6IEFwcE1ldGFkYXRhRmxhZ3MsIGFwcE1ldGFkYXRhOiBBcHBNZXRhZGF0YSkge1xuICAgICAgICBjb25zdCBuYXZpZ2F0aW9uID0gYXBwTWV0YWRhdGE/Lm5hdmlnYXRpb24gPz8ge307XG4gICAgICAgIGlmICghZW1wdHlPYmplY3QobmF2aWdhdGlvbikpIHtcbiAgICAgICAgICAgIGN1cnJlbnRNZXRhZGF0YS5uYXZpZ2F0aW9uID0gdHJ1ZTtcbiAgICAgICAgICAgIHRoaXMubmF2aWdhdGlvbi5zZXQobmF2aWdhdGlvbik7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgc2V0VGhlbWVJbWFnZXMoY3VycmVudE1ldGFkYXRhOiBBcHBNZXRhZGF0YUZsYWdzLCBhcHBNZXRhZGF0YTogQXBwTWV0YWRhdGEpIHtcbiAgICAgICAgY29uc3QgdGhlbWVJbWFnZXMgPSBhcHBNZXRhZGF0YT8udGhlbWVJbWFnZXMgPz8ge307XG4gICAgICAgIGNvbnN0IGltYWdlcyA9IHRoZW1lSW1hZ2VzPy5pdGVtcyA/PyB7fTtcbiAgICAgICAgaWYgKCFlbXB0eU9iamVjdCh0aGVtZUltYWdlcykgJiYgIWVtcHR5T2JqZWN0KGltYWdlcykpIHtcbiAgICAgICAgICAgIGN1cnJlbnRNZXRhZGF0YS50aGVtZUltYWdlcyA9IHRydWU7XG4gICAgICAgICAgICBjb25zdCB0aGVtZSA9IHRoZW1lSW1hZ2VzLmlkO1xuICAgICAgICAgICAgdGhpcy50aGVtZUltYWdlcy5zZXQodGhlbWUsIGltYWdlcyk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgc2V0UHJlZmVyZW5jZXMoY3VycmVudE1ldGFkYXRhOiBBcHBNZXRhZGF0YUZsYWdzLCBhcHBNZXRhZGF0YTogQXBwTWV0YWRhdGEpIHtcbiAgICAgICAgY29uc3QgcHJlZnMgPSBhcHBNZXRhZGF0YT8udXNlclByZWZlcmVuY2VzID8/IHt9O1xuICAgICAgICBpZiAoIWVtcHR5T2JqZWN0KHByZWZzKSkge1xuICAgICAgICAgICAgY3VycmVudE1ldGFkYXRhLnVzZXJQcmVmZXJlbmNlcyA9IHRydWU7XG4gICAgICAgICAgICBjb25zdCB1c2VyUHJlZmVyZW5jZXMgPSB0aGlzLm1hcFByZWZlcmVuY2VzKHByZWZzKTtcbiAgICAgICAgICAgIHRoaXMucHJlZmVyZW5jZXMuc2V0KHVzZXJQcmVmZXJlbmNlcyk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBwcm90ZWN0ZWQgc2V0Q29uZmlnKGN1cnJlbnRNZXRhZGF0YTogQXBwTWV0YWRhdGFGbGFncywgYXBwTWV0YWRhdGE6IEFwcE1ldGFkYXRhKSB7XG4gICAgICAgIGNvbnN0IGNvbmZpZ3MgPSBhcHBNZXRhZGF0YT8uc3lzdGVtQ29uZmlnID8/IHt9O1xuICAgICAgICBpZiAoIWVtcHR5T2JqZWN0KGNvbmZpZ3MpKSB7XG4gICAgICAgICAgICBjdXJyZW50TWV0YWRhdGEuc3lzdGVtQ29uZmlnID0gdHJ1ZTtcbiAgICAgICAgICAgIHRoaXMuY29uZmlnLnNldChjb25maWdzKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIHByb3RlY3RlZCBzZXRBZG1pbk1ldGFkYXRhKGN1cnJlbnRNZXRhZGF0YTogQXBwTWV0YWRhdGFGbGFncywgYXBwTWV0YWRhdGE6IEFwcE1ldGFkYXRhKSB7XG4gICAgICAgIGNvbnN0IGFkbWluTWV0YWRhdGEgPSBhcHBNZXRhZGF0YT8uYWRtaW5NZXRhZGF0YSA/PyB7fTtcbiAgICAgICAgaWYgKCFlbXB0eU9iamVjdChhZG1pbk1ldGFkYXRhKSkge1xuICAgICAgICAgICAgY3VycmVudE1ldGFkYXRhLmFkbWluTWV0YWRhdGEgPSB0cnVlO1xuICAgICAgICAgICAgdGhpcy5hZG1pbk1ldGFkYXRhU3RvcmUuc2V0KGFkbWluTWV0YWRhdGEpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIHNldEdsb2JhbFJlY2VudGx5Vmlld2VkKGN1cnJlbnRNZXRhZGF0YTogQXBwTWV0YWRhdGFGbGFncywgYXBwTWV0YWRhdGE6IEFwcE1ldGFkYXRhKSB7XG4gICAgICAgIGNvbnN0IGdsb2JhbFJlY2VudGx5Vmlld2VkID0gYXBwTWV0YWRhdGE/Lmdsb2JhbFJlY2VudGx5Vmlld2VkID8/IFtdO1xuICAgICAgICBpZiAoZ2xvYmFsUmVjZW50bHlWaWV3ZWQubGVuZ3RoKSB7XG4gICAgICAgICAgICBjdXJyZW50TWV0YWRhdGEuZ2xvYmFsUmVjZW50bHlWaWV3ZWQgPSB0cnVlO1xuICAgICAgICAgICAgdGhpcy5nbG9iYWxSZWNlbnRseVZpZXdlZFN0b3JlLnNldChnbG9iYWxSZWNlbnRseVZpZXdlZCk7XG4gICAgICAgIH0gZWxzZSBpZihhcHBNZXRhZGF0YT8uZ2xvYmFsUmVjZW50bHlWaWV3ZWQpIHtcbiAgICAgICAgICAgICAgICB0aGlzLmdsb2JhbFJlY2VudGx5Vmlld2VkU3RvcmUuc2V0KGdsb2JhbFJlY2VudGx5Vmlld2VkKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIHByb3RlY3RlZCBtYXBQcmVmZXJlbmNlcyhwcmVmZXJlbmNlczogYW55KTogVXNlclByZWZlcmVuY2VNYXAge1xuICAgICAgICBjb25zdCB1c2VyUHJlZmVyZW5jZXM6IFVzZXJQcmVmZXJlbmNlTWFwID0ge307XG4gICAgICAgIE9iamVjdC5rZXlzKHByZWZlcmVuY2VzKS5mb3JFYWNoKChwcmVmS2V5KSA9PiB7XG5cbiAgICAgICAgICAgIGlmICghcHJlZmVyZW5jZXNbcHJlZktleV0uaXRlbXMpIHtcbiAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIE9iamVjdC5rZXlzKHByZWZlcmVuY2VzW3ByZWZLZXldLml0ZW1zKS5mb3JFYWNoKGtleSA9PiB7XG4gICAgICAgICAgICAgICAgdXNlclByZWZlcmVuY2VzW2tleV0gPSBwcmVmZXJlbmNlc1twcmVmS2V5XS5pdGVtc1trZXldO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gdXNlclByZWZlcmVuY2VzO1xuICAgIH1cbn1cbiJdfQ==