/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { isEmpty } from 'lodash-es';
import { Injectable } from '@angular/core';
import { emptyObject, isVoid } from 'common';
import { map, take, tap } from 'rxjs/operators';
import { BehaviorSubject, combineLatestWith } from 'rxjs';
import { MessageService } from '../../../../services/message/message.service';
import { SaveFilterStoreFactory } from '../saved-filter/saved-filter.store.factory';
import * as i0 from "@angular/core";
import * as i1 from "../../../../services/message/message.service";
import * as i2 from "../saved-filter/saved-filter.store.factory";
class ListFilterStore {
    constructor(message, savedFilterStoreFactory) {
        this.message = message;
        this.savedFilterStoreFactory = savedFilterStoreFactory;
        this.panelMode = 'closable';
        this.mode = 'filter';
        this.gridButtons = null;
        this.fields = [];
        this.special = [];
        this.savedFilters = [];
        this.subs = [];
        this.filterStore = savedFilterStoreFactory.create();
    }
    init(config) {
        this.config = config;
        this.initSearchFields();
        this.initConfigUpdatesSubscription();
        this.vm$ = this.filterStore.stagingRecord$.pipe(map(stagingRecord => [stagingRecord]), tap(([savedFilter]) => {
            this.reset();
            this.splitCriteriaFields(savedFilter);
            this.initDisplayFields();
            if (this.filterStore.getMode() !== 'detail') {
                this.initGridButtons();
            }
            this.initHeaderButtons();
            this.initMyFiltersButton(this.savedFilters);
        }));
        if (this.config.panelMode) {
            this.panelMode = this.config.panelMode;
        }
        this.collapse = new BehaviorSubject(false);
        this.isCollapsed$ = this.collapse.asObservable();
        if (!isVoid(this.config.isCollapsed)) {
            this.collapse.next(this.config.isCollapsed);
        }
        this.reset();
        this.initGridButtons();
        this.initHeaderButtons();
    }
    clear() {
        this.subs.forEach(sub => sub.unsubscribe());
        this.reset();
        this.filterStore.clear();
        this.filterStore = null;
        this.collapse.unsubscribe();
    }
    clearAuthBased() {
        this.clear();
    }
    /**
     * Reset criteria
     */
    reset() {
        this.fields = [];
        this.special = [];
    }
    /**
     * Apply current filter values
     */
    applyFilter() {
        this.filterStore.validateCriteria().pipe(take(1)).subscribe(valid => {
            if (valid) {
                if (this.config.panelMode === 'collapsible' && this.config.collapseOnSearch) {
                    this.collapse.next(true);
                }
                this.config.onSearch();
                this.config.updateFilter(this.filterStore.getBaseRecord());
                return;
            }
            this.message.addWarningMessageByKey('LBL_VALIDATION_ERRORS');
        });
    }
    /**
     * Clear the current filter
     */
    clearFilter() {
        this.config.resetFilter(false);
    }
    /**
     * Subscribe to config updates.
     * Each time the filter or metadata changes we need to:
     * - Reset the view state
     * - Re-initialize the filter subscription
     */
    initConfigUpdatesSubscription() {
        this.subs.push(this.config.filter$.pipe(combineLatestWith(this.config.searchFields$), tap(([filter, searchFields]) => {
            this.reset();
            let mode = 'edit';
            const isReadOnly = filter.readonly ?? false;
            if (isReadOnly) {
                mode = 'detail';
                this.mode = mode;
            }
            this.filterStore.initRecord(this.config.module, filter, searchFields, this.config.listFields, mode);
        })).subscribe());
        this.subs.push(this.config.savedFilters$.pipe(map(value => [value]), tap(([filters]) => {
            this.savedFilters = filters;
            this.initMyFiltersButton(filters);
        })).subscribe());
    }
    /**
     * Split fields per slots
     *
     * @param {object} savedFilter to use
     */
    splitCriteriaFields(savedFilter) {
        if (!savedFilter || !savedFilter.criteriaFields) {
            return;
        }
        Object.entries(savedFilter.criteriaFields).forEach(([key, field]) => {
            const name = field.name || key;
            if (name.includes('_only')) {
                this.special.push(field);
                return;
            }
            if (!field.vardefBased) {
                this.fields.push(field);
                return;
            }
            if (field.readonly === true) {
                return;
            }
            const filters = savedFilter?.criteria?.filters ?? {};
            const fieldFilter = (filters[key] ?? {});
            if (!isEmpty(fieldFilter?.operator) && field.display === 'none') {
                field.display = 'default';
            }
            this.fields.push(field);
        });
    }
    initSearchFields() {
        this.subs.push(this.config.searchFields$.subscribe(searchFields => {
            this.searchFields = searchFields;
        }));
    }
    initDisplayFields() {
        if (!this.searchFields || emptyObject(this.searchFields) || !this.fields) {
            this.displayFields = [];
        }
        const fields = [];
        this.fields.forEach(field => {
            const name = field.name;
            if (field.display === 'none' || field.source === 'groupField') {
                return;
            }
            if (!this.searchFields[name]) {
                field.readonly = true;
            }
            fields.push(field);
        });
        this.displayFields = fields;
    }
    /**
     * Initialize grid buttons
     */
    initGridButtons() {
        this.gridButtons = [
            {
                id: 'clear',
                labelKey: 'LBL_CLEAR_BUTTON_LABEL',
                klass: ['clear-filters-button', 'btn', 'btn-outline-danger', 'btn-sm'],
                onClick: this.clearFilter.bind(this)
            },
            {
                id: 'search',
                labelKey: 'LBL_SEARCH_BUTTON_LABEL',
                klass: ['filter-button', 'btn', 'btn-danger', 'btn-sm'],
                onClick: this.applyFilter.bind(this)
            }
        ];
    }
    /**
     * Initialize header buttons
     */
    initHeaderButtons() {
        this.closeButton = {
            onClick: () => {
                this.config.onClose();
            }
        };
    }
    initMyFiltersButton(filters) {
        if (!filters || filters.length < 1) {
            this.myFilterButton = null;
            return;
        }
        const button = {
            wrapperKlass: ['filter-select'],
            labelKey: 'LBL_SAVED_FILTER_SHORTCUT',
            klass: ['btn', 'btn-outline-light', 'btn-sm'],
            items: [],
        };
        const currentKey = this.filterStore.getRecordId();
        filters.forEach(filter => {
            const item = {
                label: filter.attributes.name,
                onClick: () => {
                    this.config.setOpenFilter(filter);
                }
            };
            if (filter.key === currentKey) {
                button.label = filter.attributes.name;
                button.labelKey = '';
                item.icon = 'filter';
                item.iconKlass = 'small';
                item.klass = 'active';
            }
            button.items.push(item);
        });
        this.myFilterButton = button;
    }
    static { this.ɵfac = function ListFilterStore_Factory(t) { return new (t || ListFilterStore)(i0.ɵɵinject(i1.MessageService), i0.ɵɵinject(i2.SaveFilterStoreFactory)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ListFilterStore, factory: ListFilterStore.ɵfac }); }
}
export { ListFilterStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ListFilterStore, [{
        type: Injectable
    }], function () { return [{ type: i1.MessageService }, { type: i2.SaveFilterStoreFactory }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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