/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { deepClone } from 'common';
import { BehaviorSubject, throwError } from 'rxjs';
import { catchError, distinctUntilChanged, filter, map, shareReplay, startWith, take, tap } from 'rxjs/operators';
import { signal } from "@angular/core";
const initialState = {
    id: '',
    type: '',
    module: '',
    attributes: {},
    acls: []
};
export class RecordStore {
    constructor(definitions$, metadata$, recordSaveGQL, recordFetchGQL, message, recordManager, recordMappers) {
        this.definitions$ = definitions$;
        this.metadata$ = metadata$;
        this.recordSaveGQL = recordSaveGQL;
        this.recordFetchGQL = recordFetchGQL;
        this.message = message;
        this.recordManager = recordManager;
        this.recordMappers = recordMappers;
        this.initFieldDefaults = false;
        this.fieldDefaultsInitialized = false;
        this.cache$ = null;
        this.internalState = deepClone(initialState);
        this.stagingState = deepClone(initialState);
        this.store = new BehaviorSubject(this.internalState);
        this.staging = new BehaviorSubject(this.stagingState);
        this.definitions = [];
        this.metadata = null;
        this.subs = [];
        this.fieldsMetadata = {
            fields: [
                '_id',
                'id',
                'attributes',
                'module',
                'type',
                'acls',
                'favorite'
            ]
        };
        this.state$ = this.store.asObservable().pipe(distinctUntilChanged());
        this.staging$ = this.staging.asObservable();
        this.subs.push(this.state$.subscribe(record => {
            this.updateStaging(record);
        }));
        this.subs.push(definitions$.subscribe(definitions => {
            this.definitions = definitions;
            this.init(this.internalState);
        }));
        if (metadata$) {
            this.subs.push(metadata$.subscribe(metadata => {
                this.setMetadata(metadata);
            }));
        }
    }
    init(record, initDefaultValues = false) {
        const newRecord = {
            ...record,
        };
        this.initFieldDefaults = initDefaultValues;
        this.initRecord(newRecord);
        this.updateState(newRecord);
    }
    getStaging() {
        if (!this.stagingState) {
            return null;
        }
        return this.stagingState;
    }
    setStaging(record) {
        this.initRecord(record);
        this.staging.next(this.stagingState = record);
    }
    setRecord(record, initDefaultValues = false) {
        this.initFieldDefaults = initDefaultValues;
        this.initRecord(record);
        this.updateState(record);
    }
    save() {
        this.mapStagingFields();
        return this.recordSaveGQL.save(this.stagingState).pipe(tap((record) => {
            this.initRecord(record);
            this.updateState(record);
        }));
    }
    validate() {
        this.stagingState.formGroup.markAllAsTouched();
        this.stagingState?.validationTriggered.set(true);
        return this.stagingState.formGroup.statusChanges.pipe(startWith(this.stagingState.formGroup.status), filter(status => status !== 'PENDING'), take(1), map(status => status === 'VALID'));
    }
    resetStaging() {
        this.updateStaging(this.internalState);
    }
    destroy() {
        this.subs.forEach(sub => sub.unsubscribe());
    }
    /**
     * Get record
     *
     * @returns {object} Record
     */
    getBaseRecord() {
        if (!this.internalState) {
            return null;
        }
        const baseRecord = {
            id: this.internalState.id,
            type: this.internalState.type,
            module: this.internalState.module,
            attributes: this.internalState.attributes,
            acls: this.internalState.acls
        };
        return deepClone(baseRecord);
    }
    /**
     * Get record
     *
     * @returns {object} Record
     */
    getBaseStaging() {
        if (!this.stagingState) {
            return null;
        }
        this.mapStagingFields();
        const baseRecord = {
            id: this.stagingState.id,
            type: this.stagingState.type,
            module: this.stagingState.module,
            attributes: this.stagingState.attributes,
            acls: this.stagingState.acls
        };
        return deepClone(baseRecord);
    }
    /**
     * Extract base record
     *
     * @returns {object} Record
     */
    extractBaseRecord(record) {
        const { fields, formGroup, ...base } = record;
        return { ...base };
    }
    /**
     * Is staging record dirty
     *
     * @returns {object} Record
     */
    isDirty() {
        if (!this.stagingState || !this.stagingState.formGroup) {
            return false;
        }
        return this.stagingState.formGroup.dirty;
    }
    /**
     * Get record cached Observable or call the backend
     *
     * @param {string} module to use
     * @param {string} recordId to use
     * @param {boolean} useCache if to use cache
     * @returns {object} Observable<any>
     */
    retrieveRecord(module, recordId, useCache = true) {
        if (this.cache$ == null || useCache === false) {
            this.cache$ = this.fetch(module, recordId).pipe(tap(record => this.init(record)), shareReplay(1));
        }
        return this.cache$;
    }
    /**
     * Internal API
     */
    /**
     * Update the state
     *
     * @param {object} state to set
     */
    updateState(state) {
        this.store.next(this.internalState = state);
    }
    /**
     * Update the staging
     *
     * @param {object} state to set
     */
    updateStaging(state) {
        const newState = deepClone(this.extractBaseRecord(state));
        this.initRecord(newState, this.initFieldDefaults);
        this.staging.next(this.stagingState = newState);
    }
    /**
     * Map staging fields
     */
    mapStagingFields() {
        const mappers = this.recordMappers.get(this.stagingState.module);
        Object.keys(mappers).forEach(key => {
            const mapper = mappers[key];
            mapper.map(this.stagingState);
        });
    }
    /**
     * Init record fields
     *
     * @param {object} record Record
     * @param {boolean} initDefaultValues
     */
    initRecord(record, initDefaultValues = false) {
        if (this.metadata) {
            record.metadata = this.metadata;
        }
        if (!record?.validationTriggered) {
            record.validationTriggered = signal(false);
        }
        if (record.module && this.definitions && this.definitions.length > 0) {
            record.fields = this.recordManager.initFields(record, this.definitions);
        }
        if (initDefaultValues) {
            this.recordManager.initFieldDefaults(record);
            this.fieldDefaultsInitialized = true;
        }
    }
    /**
     * Fetch the record from the backend
     *
     * @param {string} module to use
     * @param {string} recordID to use
     * @returns {object} Observable<any>
     */
    fetch(module, recordID) {
        return this.recordFetchGQL.fetch(module, recordID, this.fieldsMetadata)
            .pipe(map(({ data }) => {
            const record = {
                type: '',
                module: '',
                attributes: {},
                acls: []
            };
            if (!data) {
                return record;
            }
            const id = data.record.attributes.id;
            if (!id) {
                this.message.addDangerMessageByKey('LBL_RECORD_DOES_NOT_EXIST');
                return record;
            }
            record.id = id;
            record.module = module;
            record.type = data.record.attributes && data.record.attributes.object_name;
            record.attributes = data.record.attributes;
            record.acls = data.record.acls;
            record.favorite = data?.record?.favorite ?? false;
            return record;
        }), catchError(err => throwError(err)));
    }
    setMetadata(metadata) {
        if (!metadata) {
            return;
        }
        this.metadata = metadata;
    }
}
//# sourceMappingURL=data:application/json;base64,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