/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { Router } from '@angular/router';
import { HttpClient, HttpHeaders, HttpParams } from '@angular/common/http';
import { BehaviorSubject, throwError } from 'rxjs';
import { catchError, distinctUntilChanged, filter, finalize, take } from 'rxjs/operators';
import { isEmptyString, isTrue } from 'common';
import { MessageService } from '../message/message.service';
import { StateManager } from '../../store/state-manager';
import { LanguageStore } from '../../store/language/language.store';
import { AppStateStore } from '../../store/app-state/app-state.store';
import { LocalStorageService } from '../local-storage/local-storage.service';
import { SystemConfigStore } from '../../store/system-config/system-config.store';
import { BaseRouteService } from "../base-route/base-route.service";
import { NotificationStore } from '../../store/notification/notification.store';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "@angular/router";
import * as i3 from "../message/message.service";
import * as i4 from "../../store/state-manager";
import * as i5 from "../../store/language/language.store";
import * as i6 from "../../store/app-state/app-state.store";
import * as i7 from "../local-storage/local-storage.service";
import * as i8 from "../../store/system-config/system-config.store";
import * as i9 from "../base-route/base-route.service";
import * as i10 from "../../store/notification/notification.store";
class AuthService {
    constructor(http, router, message, stateManager, languageStore, appStateStore, localStorage, configs, baseRoute, notificationStore) {
        this.http = http;
        this.router = router;
        this.message = message;
        this.stateManager = stateManager;
        this.languageStore = languageStore;
        this.appStateStore = appStateStore;
        this.localStorage = localStorage;
        this.configs = configs;
        this.baseRoute = baseRoute;
        this.notificationStore = notificationStore;
        this.isUserLoggedIn = new BehaviorSubject(false);
        this.currentUserSubject = new BehaviorSubject({});
        this.currentUser$ = this.currentUserSubject.asObservable().pipe(distinctUntilChanged());
    }
    isLoggedIn() {
        return this.isUserLoggedIn.value;
    }
    getCurrentUser() {
        return this.currentUserSubject.value;
    }
    setCurrentUser(data) {
        this.appStateStore.setCurrentUser(data);
        this.currentUserSubject.next(data);
        this.isUserLoggedIn.next(true);
    }
    doLogin(username, password, onSuccess, onError) {
        let loginUrl = 'login';
        loginUrl = this.baseRoute.appendNativeAuth(loginUrl);
        loginUrl = this.baseRoute.calculateRoute(loginUrl);
        const headers = new HttpHeaders({
            'Content-Type': 'application/json',
        });
        return this.http.post(loginUrl, {
            username,
            password
        }, { headers }).subscribe((response) => {
            if (this.baseRoute.isNativeAuth()) {
                window.location.href = this.baseRoute.removeNativeAuth();
            }
            this.appStateStore.updateInitialAppLoading(true);
            onSuccess(response);
            this.isUserLoggedIn.next(true);
            this.setCurrentUser(response);
            this.notificationStore.enableNotifications();
            this.notificationStore.refreshNotifications();
        }, (error) => {
            onError(error);
        });
    }
    /**
     * Logout user
     *
     * @param {string} messageKey of message to display
     * @param {boolean} redirect to home
     */
    logout(messageKey = 'LBL_LOGOUT_SUCCESS', redirect = true) {
        this.appStateStore.updateLoading('logout', true, false);
        const logoutConfig = this.configs.getConfigValue('logout') ?? [];
        let logoutUrl = (logoutConfig?.path ?? 'logout');
        let redirectLogout = isTrue(logoutConfig?.redirect ?? false);
        const afterLogoutPath = (logoutConfig?.after_logout_path ?? './');
        if (this.baseRoute.isNativeAuth()) {
            logoutUrl = this.baseRoute.getNativeOutLogoutUrl();
            redirectLogout = false;
        }
        logoutUrl = this.baseRoute.calculateRoute(logoutUrl);
        const body = new HttpParams();
        const headers = new HttpHeaders().set('Content-Type', 'text/plain; charset=utf-8');
        if (this.appStateStore.getActiveRequests() < 1) {
            this.callLogout(logoutUrl, body, headers, redirect, messageKey, redirectLogout, afterLogoutPath);
        }
        else {
            this.appStateStore.activeRequests$.pipe(filter(value => value < 1), take(1)).subscribe(() => {
                this.callLogout(logoutUrl, body, headers, redirect, messageKey, redirectLogout, afterLogoutPath);
            });
        }
    }
    /**
     * Call logout
     * @param logoutUrl
     * @param body
     * @param headers
     * @param redirect
     * @param messageKey
     * @param redirectLogout
     * @protected
     */
    callLogout(logoutUrl, body, headers, redirect, messageKey, redirectLogout, afterLogoutPath) {
        this.resetState();
        if (redirectLogout) {
            window.location.href = logoutUrl;
            return;
        }
        this.http.post(logoutUrl, body.toString(), { headers, responseType: 'text' })
            .pipe(take(1), catchError(err => {
            this.message.log('Logout failed');
            return throwError(err);
        }), finalize(() => {
            this.appStateStore.updateInitialAppLoading(true);
            this.appStateStore.updateLoading('logout', false, false);
            this.appStateStore.setCurrentUser(null);
            this.stateManager.clearAuthBased();
            this.configs.clear();
            if (redirect === true) {
                window.location.href = afterLogoutPath;
            }
        }))
            .subscribe(() => {
            this.message.log('Logout success');
            if (!isEmptyString(messageKey)) {
                this.message.addSuccessMessageByKey(messageKey);
            }
        }, () => {
            this.message.log('Error on logout');
            if (!isEmptyString(messageKey)) {
                this.message.addSuccessMessageByKey(messageKey);
            }
        });
    }
    /**
     * On logout state reset
     */
    resetState() {
        this.stateManager.clearAuthBased();
        this.localStorage.clear();
        this.isUserLoggedIn.next(false);
    }
    /**
     * Fetch session status from backend
     *
     * @returns {{}} Observable<SessionStatus>
     */
    fetchSessionStatus() {
        let url = 'session-status';
        url = this.baseRoute.appendNativeAuth(url);
        url = this.baseRoute.calculateRoute(url);
        const headers = new HttpHeaders().set('Content-Type', 'text/plain; charset=utf-8');
        return this.http.get(url, { headers });
    }
    /**
     * Get route for session expired handling
     * @return string
     */
    getSessionExpiredRoute() {
        const sessionExpiredConfig = this.configs.getConfigValue('session-expired') ?? [];
        return (sessionExpiredConfig?.path ?? 'Login');
    }
    /**
     * Handle invalid session on request
     * @return boolean
     */
    handleInvalidSession(message) {
        const redirect = this.sessionExpiredRedirect();
        if (redirect) {
            this.handleSessionExpiredRedirect();
            return;
        }
        this.logout(message);
    }
    /**
     * Redirect to route configured for session expiry
     */
    handleSessionExpiredRedirect() {
        window.location.href = this.getSessionExpiredRoute();
    }
    /**
     * Is to re-direct on session expiry
     * @return boolean
     */
    sessionExpiredRedirect() {
        const sessionExpiredConfig = this.configs.getConfigValue('session-expired') ?? [];
        return isTrue(sessionExpiredConfig?.redirect ?? false);
    }
    static { this.ɵfac = function AuthService_Factory(t) { return new (t || AuthService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.Router), i0.ɵɵinject(i3.MessageService), i0.ɵɵinject(i4.StateManager), i0.ɵɵinject(i5.LanguageStore), i0.ɵɵinject(i6.AppStateStore), i0.ɵɵinject(i7.LocalStorageService), i0.ɵɵinject(i8.SystemConfigStore), i0.ɵɵinject(i9.BaseRouteService), i0.ɵɵinject(i10.NotificationStore)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: AuthService, factory: AuthService.ɵfac, providedIn: 'root' }); }
}
export { AuthService };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(AuthService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], function () { return [{ type: i1.HttpClient }, { type: i2.Router }, { type: i3.MessageService }, { type: i4.StateManager }, { type: i5.LanguageStore }, { type: i6.AppStateStore }, { type: i7.LocalStorageService }, { type: i8.SystemConfigStore }, { type: i9.BaseRouteService }, { type: i10.NotificationStore }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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