/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { ValidationManager } from '../validation/validation.manager';
import { DataTypeFormatter } from '../../formatters/data-type.formatter.service';
import { isFalse, isTrue } from 'common';
import { UntypedFormArray, UntypedFormControl } from '@angular/forms';
import get from 'lodash-es/get';
import { isEmpty, merge } from 'lodash-es';
import { FieldObjectRegistry } from "./field-object-type.registry";
import * as i0 from "@angular/core";
import * as i1 from "../validation/validation.manager";
import * as i2 from "../../formatters/data-type.formatter.service";
import * as i3 from "./field-object-type.registry";
class FieldBuilder {
    constructor(validationManager, typeFormatter, fieldRegistry) {
        this.validationManager = validationManager;
        this.typeFormatter = typeFormatter;
        this.fieldRegistry = fieldRegistry;
    }
    /**
     * Build field
     *
     * @param {object} record Record
     * @param {object} viewField ViewFieldDefinition
     * @param {object} language LanguageStore
     * @returns {object}Field
     */
    buildField(record, viewField, language = null) {
        const module = record?.module ?? '';
        let definition = viewField?.fieldDefinition ?? {};
        const multiModuleDefinitions = viewField?.multiModuleDefinitions ?? {};
        const currentModuleDefinitions = multiModuleDefinitions[module] ?? {};
        if (!isEmpty(currentModuleDefinitions)) {
            definition = currentModuleDefinitions;
        }
        const { value, valueList, valueObject } = this.parseValue(viewField, definition, record);
        const { validators, asyncValidators, itemFormArraySaveValidators } = this.getSaveValidators(record, viewField);
        const field = this.setupField(record.module, viewField, value, valueList, valueObject, record, definition, validators, asyncValidators, language);
        field.itemFormArraySaveValidators = itemFormArraySaveValidators;
        return field;
    }
    getFieldLabel(label, module, language) {
        const languages = language.getLanguageStrings();
        return language.getFieldLabel(label, module, languages);
    }
    /**
     * Parse value from record
     *
     * @param {object} viewField ViewFieldDefinition
     * @param {object} definition FieldDefinition
     * @param {object} record Record
     * @returns {object} value object
     */
    parseValue(viewField, definition, record) {
        const type = (viewField && viewField.type) || '';
        const source = (definition && definition.source) || '';
        const rname = (definition && definition.rname) || 'name';
        const viewName = viewField.name || '';
        let value = null;
        let valueList = null;
        if (!viewName || (!record.attributes[viewName] && !isFalse(record.attributes[viewName]))) {
            value = '';
        }
        else if (type === 'relate' && source === 'non-db' && rname !== '') {
            value = record.attributes[viewName][rname];
            const valueObject = record.attributes[viewName];
            return { value, valueList, valueObject };
        }
        else {
            value = record.attributes[viewName];
        }
        if (type === 'line-items') {
            return { value: null, valueList };
        }
        if (Array.isArray(value)) {
            valueList = value;
            value = null;
        }
        return { value, valueList };
    }
    /**
     * Build and initialize field object
     *
     * @param {string} module to use
     * @param {object} viewField ViewFieldDefinition
     * @param {string} value string
     * @param {[]} valueList string[]
     * @param {} valueObject value object
     * @param {object} record Record
     * @param {object} definition FieldDefinition
     * @param {[]} validators ValidatorFn[]
     * @param {[]} asyncValidators AsyncValidatorFn[]
     * @param {object} language LanguageStore
     * @returns {object} BaseField
     */
    setupField(module, viewField, value, valueList, valueObject, record, definition, validators, asyncValidators, language) {
        const metadata = merge(definition?.metadata ?? {}, viewField?.metadata ?? {});
        const formattedValue = this.typeFormatter.toUserFormat(viewField.type, value, { mode: 'edit', metadata });
        if (viewField.link) {
            metadata.link = viewField.link;
        }
        const type = viewField.type || definition.type;
        const FieldObjectType = this.fieldRegistry.get(module, type);
        const field = new FieldObjectType();
        field.type = type;
        field.name = viewField.name || definition.name || '';
        field.vardefBased = viewField?.vardefBased ?? definition?.vardefBased ?? false;
        field.readonly = isTrue(viewField.readonly) || isTrue(definition.readonly) || false;
        field.display = (viewField.display || definition.display || 'default');
        field.defaultDisplay = field.display;
        if (field.defaultDisplay === 'default') {
            field.defaultDisplay = 'show';
        }
        field.value = value;
        field.metadata = metadata;
        field.definition = definition;
        if (viewField?.lineItems) {
            field.definition.lineItems = viewField.lineItems;
        }
        field.labelKey = viewField.label || definition.vname || '';
        field.dynamicLabelKey = viewField.dynamicLabelKey || definition.dynamicLabelKey || '';
        const defaultValue = viewField?.defaultValue ?? definition?.default ?? definition?.defaultValue ?? null;
        if (defaultValue) {
            field.default = defaultValue;
        }
        field.defaultValueModes = viewField?.defaultValueModes ?? definition?.defaultValueModes ?? [];
        field.validators = validators;
        field.asyncValidators = asyncValidators;
        if (field.type === 'line-items') {
            field.valueObjectArray = record.attributes[field.name];
            field.itemFormArray = new UntypedFormArray([]);
            field.formControl = new UntypedFormControl(formattedValue);
        }
        else {
            field.formControl = new UntypedFormControl(formattedValue);
        }
        field.attributes = {};
        field.source = 'field';
        field.logic = viewField.logic || definition.logic || null;
        field.displayLogic = viewField.displayLogic || definition.displayLogic || null;
        const fieldDependencies = {};
        const attributeDependencies = {};
        this.addFieldDependencies(field.logic, fieldDependencies, attributeDependencies, 'logic');
        this.addFieldDependencies(field.displayLogic, fieldDependencies, attributeDependencies, 'displayLogic');
        field.attributeDependencies = Object.keys(attributeDependencies).map(key => attributeDependencies[key]);
        field.fieldDependencies = fieldDependencies;
        if (valueList) {
            field.valueList = valueList;
        }
        if (valueObject) {
            field.valueObject = valueObject;
        }
        if (language) {
            field.label = this.getFieldLabel(viewField.label, module, language);
        }
        if (!field.labelKey && viewField.label) {
            field.labelKey = viewField.label;
        }
        return field;
    }
    addFieldDependencies(config, fieldDependencies, attributeDependencies, type) {
        if (config && Object.keys(config).length) {
            Object.keys(config).forEach(logicKey => {
                const entry = config[logicKey] || {};
                if (!entry.params) {
                    return;
                }
                if (entry.params && entry.params.attributeDependencies) {
                    entry.params.attributeDependencies.forEach(dependency => {
                        const dependencyKey = dependency.field + '.' + dependency.attribute;
                        attributeDependencies[dependencyKey] = dependency;
                    });
                }
                if (entry.params && entry.params.fieldDependencies) {
                    entry.params.fieldDependencies.forEach(dependency => {
                        const fieldDependency = fieldDependencies[dependency] ?? {};
                        const types = fieldDependency['types'] ?? [];
                        types.push(type);
                        fieldDependencies[dependency] = {
                            field: dependency,
                            type: types
                        };
                    });
                }
            });
        }
    }
    /**
     * Get save validators for the given field definition
     *
     * @param {object} record Record
     * @param {object} viewField ViewFieldDefinition
     * @returns {object} Validator map
     */
    getSaveValidators(record, viewField) {
        const validators = this.validationManager.getSaveValidations(record.module, viewField, record);
        const asyncValidators = this.validationManager.getAsyncSaveValidations(record.module, viewField, record);
        const itemFormArraySaveValidators = this.validationManager.getItemFormArraySaveValidations(record.module, viewField, record);
        return { validators, asyncValidators, itemFormArraySaveValidators };
    }
    /**
     * Set attribute value on parent
     *
     * @param {object} record Record
     * @param {object} field Field
     * @param {string} name String
     * @param {object} definition FieldDefinition
     * @returns any
     */
    getParentValue(record, field, name, definition) {
        const valueParent = definition.valueParent ?? 'field';
        const parent = valueParent === 'record' ? record : field;
        if (definition.valuePath) {
            return get(parent, definition.valuePath, '');
        }
        if (valueParent === 'record') {
            return get(record.attributes, name, '');
        }
        return get(field.valueObject, name, '');
    }
    static { this.ɵfac = function FieldBuilder_Factory(t) { return new (t || FieldBuilder)(i0.ɵɵinject(i1.ValidationManager), i0.ɵɵinject(i2.DataTypeFormatter), i0.ɵɵinject(i3.FieldObjectRegistry)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: FieldBuilder, factory: FieldBuilder.ɵfac, providedIn: 'root' }); }
}
export { FieldBuilder };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(FieldBuilder, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], function () { return [{ type: i1.ValidationManager }, { type: i2.DataTypeFormatter }, { type: i3.FieldObjectRegistry }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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